<?php

// modules/anna/anna.php
defined('BASEPATH') or exit('No direct script access allowed');

/*
Module Name: Anna
Description: AI-Powered Copilot Chat Assistant for Perfex CRM
Version: 1.0.0
Author: Amjad Abbas
Author URI: https://myperfexcrm.com
Requires at least: 2.3.0
*/

define('ANNA_MODULE_NAME', 'anna');
define('ANNA_MODULE_UPLOAD_FOLDER', module_dir_path(ANNA_MODULE_NAME, 'uploads'));

hooks()->add_filter('module_anna_action_links', 'anna_action_links');
function anna_action_links(array $actions): array
{
    $actions[] = '<a href="' . admin_url('settings?group=anna') . '">' . _l('view_anna_plans') . '</a>';
    return $actions;
}

/**
 * Register activation hook
 */
register_activation_hook(ANNA_MODULE_NAME, 'anna_activation_hook');

function anna_activation_hook()
{
    $CI = &get_instance();
    require_once(__DIR__ . '/install.php');

    anna_update_htaccess();
}

function anna_update_htaccess()
{
    $htaccessPath = FCPATH . '.htaccess';

    // Authorization header rules
    $authHeader = <<<HTACCESS

# Ensure Authorization headers are passed to PHP
RewriteCond %{HTTP:Authorization} .
RewriteRule ^ - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]
HTACCESS;

    // If .htaccess exists, modify it
    if (file_exists($htaccessPath)) {
        $currentContent = file_get_contents($htaccessPath);

        // Prevent duplicate entries
        if (strpos($currentContent, 'E=HTTP_AUTHORIZATION') === false) {
            // Backup .htaccess before modifying
            copy($htaccessPath, $htaccessPath . '.bak');

            // Insert authorization rules inside <IfModule mod_rewrite.c>
            $updatedContent = preg_replace('/(<IfModule mod_rewrite\.c>)/', "$1\n" . $authHeader, $currentContent, 1);

            // Ensure safe file writing
            if (is_writable($htaccessPath)) {
                file_put_contents($htaccessPath, $updatedContent);
            }
        }
    } else {
        $defaultContent = <<<HTACCESS
AddDefaultCharset utf-8
<IfModule mod_rewrite.c>
RewriteEngine on
RewriteCond \$1 !^(index\.php|resources|robots\.txt)
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.php?/\$1 [L,QSA]
</IfModule>
$authHeader
HTACCESS;

        // Ensure safe file creation
        if (is_writable(FCPATH)) {
            file_put_contents($htaccessPath, $defaultContent);
        }
    }
}

/**
 * Register language files, must be registered if the module is using languages
 */
register_language_files(ANNA_MODULE_NAME, [ANNA_MODULE_NAME]);

/**
 * Load module helper
 */
$CI = &get_instance();
$CI->load->helper(ANNA_MODULE_NAME . '/anna');

/**
 * Init module menu items in setup in admin_init hook
 */
hooks()->add_action('admin_init', 'anna_init_menu_items');
hooks()->add_action('app_admin_head', 'anna_app_admin_head');

function anna_app_admin_head()
{
    $CI = &get_instance();

    $CI->load->model(ANNA_MODULE_NAME . '/Api_model');
    $CI->load->config(ANNA_MODULE_NAME . '/rest');

    $_data = [];
    if ($CI->db->table_exists(db_prefix() . 'mobile_app_access_tokens')) {
        if (total_rows(db_prefix() . 'mobile_app_access_tokens', ['userid' => get_staff_user_id(), 'login_as' => 'staff']) == 0) {
            $CI->load->library(ANNA_MODULE_NAME . '/Authorization_Token');
            $key = $CI->Api_model->_generate_key();

            // If no key level provided, provide a generic key
            $level = 1;
            $ignore_limits = 1;

            // Insert the new key
            if ($CI->Api_model->_insert_key($key, ['level' => $level, 'ignore_limits' => $ignore_limits, 'user_id' => get_staff_user_id()])) {
                $data = [
                    'login_as' => 'staff',
                    'userid' => get_staff_user_id(),
                    'expiration_date' => date('Y-m-d H:i:s', strtotime('+1 year'))
                ];
                $data['token'] = $CI->authorization_token->generateToken($data);

                $data['expiration_date'] = to_sql_date($data['expiration_date'], true);
                $CI->db->insert(db_prefix() . 'mobile_app_access_tokens', $data);
                $insert_id = $CI->db->insert_id();
                if ($insert_id) {
                    $_data = [
                        'auth_key' => $data['token'],
                        'key'   => $key,
                        'staff'  => get_staff()
                    ];
                }
            }
        } else {
            $mobile_app_access_tokens = $CI->db->where(['userid' => get_staff_user_id(), 'login_as' => 'staff'])->get(db_prefix() . 'mobile_app_access_tokens')->row();
            $keys = $CI->db->where('user_id', get_staff_user_id())->get(db_prefix() . 'keys')->row();

            $_data = [
                'auth_key' => $mobile_app_access_tokens->token,
                'key'   => $keys->key,
                'staff'  => get_staff()
            ];
        }
    }

    if (file_exists(APP_MODULES_PATH . '/' . ANNA_MODULE_NAME . '/config/token.php')) {
        $token    = file_get_contents(APP_MODULES_PATH . '/' . ANNA_MODULE_NAME . '/config/token.php');
        $_data['token'] = $token;
        echo '<script>';
        echo 'app.anna = {
            loggedUser: ' . json_encode($_data) . '
        };';
        echo '</script>';
    }
}
/**
 * Add chat widget to all admin pages
 */
hooks()->add_action('app_admin_footer', 'anna_add_widget');

/**
 * Init Goals Module Menu Items Setup
 */
function anna_init_menu_items()
{
    $CI = &get_instance();

    if (has_permission('anna', '', 'view')) {
        $CI->app_menu->add_sidebar_menu_item('anna', [
            'name'     => _l('anna_chat'),
            'href'     => admin_url('anna/chat'),
            'position' => 35,
            'icon'     => 'fa fa-robot',
        ]);

        $CI->app_tabs->add_settings_tab('anna', [
            'name'     => _l('anna'),
            'view'     => 'anna/settings_tab',
            'position' => 100,
            'icon'     => 'fa fa-robot',
        ]);
    }
}

/**
 * Register permissions
 */
hooks()->add_filter('staff_permissions', 'anna_permissions');

function anna_permissions($permissions)
{
    $permissions['anna'] = [
        'view'   => _l('permission_view') . '(' . _l('anna') . ')',
        'create' => _l('permission_create') . '(' . _l('anna') . ')',
        'edit'   => _l('permission_edit') . '(' . _l('anna') . ')',
        'delete' => _l('permission_delete') . '(' . _l('anna') . ')',
    ];

    return $permissions;
}

/**
 * Module uninstall hook
 */
hooks()->add_action('pre_deactivate_' . ANNA_MODULE_NAME, 'anna_uninstall');

function anna_uninstall()
{
    // Add cleanup logic here if needed
}

/**
 * Add chat widget to admin footer
 */
function anna_add_widget()
{
    // Only show to users with permission
    if (!has_permission('anna', '', 'view')) {
        return;
    }

    // Don't show on the chat page itself
    $CI = &get_instance();
    if ($CI->router->fetch_class() == 'anna' && $CI->router->fetch_method() == 'chat') {
        return;
    }

    // Include the widget (show even if not configured for testing frontend features)
    echo $CI->load->view(ANNA_MODULE_NAME . '/chat_widget', [], true);
}


hooks()->add_action('app_init', 'anna_actLib');
function anna_actLib()
{
    $CI = &get_instance();
    $CI->load->library(ANNA_MODULE_NAME . '/Envatoapi');
    $envato_res = $CI->envatoapi->validatePurchase(ANNA_MODULE_NAME);
    if (!$envato_res) {
        set_alert('danger', "One of your modules failed its verification and got deactivated. Please reactivate or contact support.");
        redirect(admin_url('modules'));
    }
}

hooks()->add_action('pre_activate_module', 'anna_sidecheck');
function anna_sidecheck($module_name)
{
    if ($module_name['system_name'] == ANNA_MODULE_NAME) {
        if (!option_exists(ANNA_MODULE_NAME . '_verified') && empty(get_option(ANNA_MODULE_NAME . '_verified')) && !option_exists(ANNA_MODULE_NAME . '_verification_id') && empty(get_option(ANNA_MODULE_NAME . '_verification_id'))) {
            $CI = &get_instance();
            $data['submit_url'] = $module_name['system_name'] . '/env_ver/activate';
            $data['original_url'] = admin_url('modules/activate/' . ANNA_MODULE_NAME);
            $data['module_name'] = ANNA_MODULE_NAME;
            $data['title']       = $module_name['headers']['module_name'] . " module activation";
            echo $CI->load->view($module_name['system_name'] . '/activate', $data, true);
            exit();
        }
    }
}

hooks()->add_action('pre_deactivate_module', ANNA_MODULE_NAME . '_deregister');
function anna_deregister($module_name)
{
    if ($module_name['system_name'] == ANNA_MODULE_NAME) {
        $CI = &get_instance();
        $CI->load->library(ANNA_MODULE_NAME . '/Envatoapi');
        $CI->envatoapi->deactivateLicense(ANNA_MODULE_NAME);
        delete_option(ANNA_MODULE_NAME . "_verified");
        delete_option(ANNA_MODULE_NAME . "_verification_id");
        delete_option(ANNA_MODULE_NAME . "_last_verification");
        delete_option(ANNA_MODULE_NAME . "_expire_verification");
        if (file_exists(__DIR__ . "/config/token.php")) {
            unlink(__DIR__ . "/config/token.php");
        }
    }
}
