/**
 * Anna Simple Cross-Tab Synchronization
 * Uses localStorage and storage events for real-time cross-tab communication
 * Much simpler than SSE - no backend complexity needed!
 */

class AnnaSync {
    constructor() {
        this.currentConversationId = this.loadCurrentConversation();
        this.widgetState = this.loadWidgetState();
        this.messageHistory = this.loadMessageHistory();
        this.tabId = this.generateTabId();

        this.init();
    }

    init() {
        // Listen for storage events from other tabs
        window.addEventListener('storage', (event) => this.handleStorageEvent(event));

        // Listen for page visibility change
        document.addEventListener('visibilitychange', () => {
            if (document.visibilityState === 'visible') {
                this.syncFromStorage();
            }
        });

    }

    /**
     * Handle storage events from other tabs
     */
    handleStorageEvent(event) {
        if (!event.key || !event.key.startsWith('pc_')) return;

        try {
            let data;

            // Handle different data types for different keys
            if (event.key === 'pc_current_conversation') {
                // This is just a plain string, not JSON
                data = event.newValue;
            } else {
                // These are JSON objects
                data = event.newValue ? JSON.parse(event.newValue) : null;
            }

            switch (event.key) {
                case 'pc_widget_state':
                    this.syncWidgetState(data);
                    break;

                case 'pc_current_conversation':
                    this.currentConversationId = data;
                    break;

                case 'pc_new_message':
                    this.syncNewMessage(data);
                    break;

                case 'pc_new_conversation_created':
                    this.syncNewConversation(data);
                    break;

                case 'pc_conversation_switch':
                    this.syncConversationSwitch(data);
                    break;
            }
        } catch (e) {
        }
    }

    /**
     * Sync widget open/close state from other tabs
     */
    syncWidgetState(data) {
        if (!data) {
            return;
        }

        const widget = document.getElementById('pc-widget-chat');
        if (!widget) {
            return;
        }

        // Update widget visibility - check if widgetIsOpen exists in global scope
        const currentState = window.widgetIsOpen !== undefined ? window.widgetIsOpen : false;
        if (data.is_open !== currentState) {
            if (data.is_open) {
                widget.style.display = 'flex';
                if (window.widgetIsOpen !== undefined) window.widgetIsOpen = true;

                // Focus input when opening
                setTimeout(() => {
                    const input = document.getElementById('pc-widget-input');
                    if (input) input.focus();
                }, 100);
            } else {
                widget.style.display = 'none';
                if (window.widgetIsOpen !== undefined) window.widgetIsOpen = false;
            }

        }

        this.widgetState = data;
    }

    /**
     * Sync new message from other tabs
     */
    syncNewMessage(data) {
        // Don't sync messages from the same tab
        if (!data) {
            return;
        }

        if (data.tab_id === this.tabId) {
            return;
        }

        if (data.conversation_id !== this.currentConversationId) {
            return;
        }

        // Check if addWidgetMessage function exists
        if (typeof addWidgetMessage !== 'function') {
            return;
        }

        // Add user message if not already present
        if (data.message) {
            const alreadyDisplayed = this.isMessageAlreadyDisplayed(data.message, 'user');
            if (!alreadyDisplayed) {
                addWidgetMessage(data.message, 'user');
            }
        }

        // Add AI response if not already present
        if (data.response) {
            const alreadyDisplayed = this.isMessageAlreadyDisplayed(data.response, 'ai');
            if (!alreadyDisplayed) {
                addWidgetMessage(data.response, 'ai');
            }
        }
    }

    /**
     * Sync conversation switch from other tabs
     */
    syncConversationSwitch(data) {
        if (!data || data.conversation_id === this.currentConversationId) return;

        // Switch to the new conversation
        this.currentConversationId = data.conversation_id;
        switchWidgetConversation(data.conversation_id, false); // false = don't trigger storage event again
    }

    /**
     * Sync new conversation creation from other tabs
     */
    syncNewConversation(data) {
        if (!data) return;

        // Reload conversation list
        loadWidgetConversations();

        // Switch to new conversation if auto_switch is enabled
        if (data.auto_switch && data.conversation_id !== this.currentConversationId) {
            this.currentConversationId = data.conversation_id;
            switchWidgetConversation(data.conversation_id, false);
        }
    }

    /**
     * Check if message is already displayed (prevent duplicates)
     */
    isMessageAlreadyDisplayed(messageText, type) {
        const messages = document.querySelectorAll('.pc-widget-message');
        const lastMessage = messages[messages.length - 1];

        if (!lastMessage) return false;

        const messageContent = lastMessage.querySelector('.pc-widget-message-text');
        const isCorrectType = type === 'user' ?
            lastMessage.classList.contains('pc-widget-user-message') :
            lastMessage.classList.contains('pc-widget-ai-message');

        return isCorrectType && messageContent && messageContent.textContent.trim() === messageText.trim();
    }

    /**
     * Sync all state from storage (when tab becomes visible)
     */
    syncFromStorage() {
        // Sync widget state
        const widgetState = this.loadWidgetState();
        if (JSON.stringify(widgetState) !== JSON.stringify(this.widgetState)) {
            this.syncWidgetState(widgetState);
        }

        // Sync current conversation
        const currentConv = this.loadCurrentConversation();
        if (currentConv !== this.currentConversationId) {
            this.syncConversationSwitch({ conversation_id: currentConv });
        }
    }

    // ======================
    // Broadcasting Methods (when THIS tab makes changes)
    // ======================

    /**
     * Broadcast widget state change to other tabs
     */
    broadcastWidgetState(isOpen, isMinimized = false, position = 'bottom-right') {
        const state = {
            is_open: isOpen,
            is_minimized: isMinimized,
            position: position,
            timestamp: Date.now()
        };

        this.widgetState = state;
        localStorage.setItem('pc_widget_state', JSON.stringify(state));
    }

    /**
     * Broadcast new message to other tabs
     */
    broadcastMessage(conversationId, message, response) {
        // Update current conversation ID and sync it
        this.currentConversationId = conversationId;
        localStorage.setItem('pc_current_conversation', conversationId);

        const data = {
            conversation_id: conversationId,
            message: message,
            response: response,
            timestamp: Date.now(),
            tab_id: this.getTabId() // Add unique tab ID to prevent self-sync
        };

        localStorage.setItem('pc_new_message', JSON.stringify(data));

        // Remove after a short delay to allow other tabs to catch it
        setTimeout(() => {
            localStorage.removeItem('pc_new_message');
        }, 1000);
    }

    /**
     * Broadcast conversation switch to other tabs
     */
    broadcastConversationSwitch(conversationId, conversationTitle) {
        const data = {
            conversation_id: conversationId,
            conversation_title: conversationTitle,
            timestamp: Date.now()
        };

        this.currentConversationId = conversationId;
        localStorage.setItem('pc_current_conversation', conversationId);
        localStorage.setItem('pc_conversation_switch', JSON.stringify(data));

        // Remove the event trigger after delay
        setTimeout(() => {
            localStorage.removeItem('pc_conversation_switch');
        }, 1000);
    }

    /**
     * Broadcast new conversation creation to other tabs
     */
    broadcastNewConversation(conversationId, conversationTitle, autoSwitch = true) {
        const data = {
            conversation_id: conversationId,
            conversation_title: conversationTitle,
            auto_switch: autoSwitch,
            timestamp: Date.now()
        };

        if (autoSwitch) {
            this.currentConversationId = conversationId;
            localStorage.setItem('pc_current_conversation', conversationId);
        }

        localStorage.setItem('pc_new_conversation_created', JSON.stringify(data));

        // Remove after delay
        setTimeout(() => {
            localStorage.removeItem('pc_new_conversation_created');
        }, 1000);
    }

    // ======================
    // Storage Helper Methods
    // ======================

    loadWidgetState() {
        try {
            const state = localStorage.getItem('pc_widget_state');
            return state ? JSON.parse(state) : {
                is_open: false,
                is_minimized: false,
                position: 'bottom-right'
            };
        } catch (e) {
            return { is_open: false, is_minimized: false, position: 'bottom-right' };
        }
    }

    loadCurrentConversation() {
        return localStorage.getItem('pc_current_conversation') || null;
    }

    loadMessageHistory() {
        try {
            const history = localStorage.getItem('pc_message_history');
            return history ? JSON.parse(history) : [];
        } catch (e) {
            return [];
        }
    }

    saveMessageHistory(message) {
        try {
            this.messageHistory.push(message);
            // Keep only last 50 messages
            if (this.messageHistory.length > 50) {
                this.messageHistory = this.messageHistory.slice(-50);
            }
            localStorage.setItem('pc_message_history', JSON.stringify(this.messageHistory));
        } catch (e) {
        }
    }

    getMessageHistory() {
        return this.messageHistory;
    }

    // ======================
    // Arrow Key Navigation for Message History
    // ======================

    setupMessageHistoryNavigation() {
        const input = document.getElementById('pc-widget-input');
        if (!input) return;

        let historyIndex = -1;
        let originalMessage = '';

        input.addEventListener('keydown', (event) => {
            const history = this.getMessageHistory();

            if (event.key === 'ArrowUp') {
                // Only handle if cursor is at the beginning or if it's a single line
                const cursorPosition = input.selectionStart;
                const textBeforeCursor = input.value.substring(0, cursorPosition);
                const lines = input.value.split('\n');
                const currentLineIndex = textBeforeCursor.split('\n').length - 1;

                // Only navigate history if cursor is on the first line
                if (currentLineIndex === 0) {
                    event.preventDefault();

                    if (historyIndex === -1) {
                        originalMessage = input.value;
                    }

                    if (historyIndex < history.length - 1) {
                        historyIndex++;
                        input.value = history[history.length - 1 - historyIndex];
                        // Move cursor to end
                        setTimeout(() => {
                            input.setSelectionRange(input.value.length, input.value.length);
                        }, 0);
                    }
                }
            } else if (event.key === 'ArrowDown') {
                // Only handle if cursor is at the end or if it's a single line
                const cursorPosition = input.selectionStart;
                const textAfterCursor = input.value.substring(cursorPosition);
                const lines = input.value.split('\n');
                const currentLineIndex = input.value.substring(0, cursorPosition).split('\n').length - 1;

                // Only navigate history if cursor is on the last line
                if (currentLineIndex === lines.length - 1) {
                    event.preventDefault();

                    if (historyIndex > 0) {
                        historyIndex--;
                        input.value = history[history.length - 1 - historyIndex];
                        // Move cursor to end
                        setTimeout(() => {
                            input.setSelectionRange(input.value.length, input.value.length);
                        }, 0);
                    } else if (historyIndex === 0) {
                        historyIndex = -1;
                        input.value = originalMessage;
                        // Move cursor to end
                        setTimeout(() => {
                            input.setSelectionRange(input.value.length, input.value.length);
                        }, 0);
                    }
                }
            } else if (event.key === 'Enter' && !event.shiftKey) {
                // Reset history navigation when sending
                historyIndex = -1;
                originalMessage = '';

                // Save message to history
                const message = input.value.trim();
                if (message) {
                    this.saveMessageHistory(message);
                }
            } else {
                // Reset navigation when user starts typing
                historyIndex = -1;
                originalMessage = '';
            }
        });
    }

    /**
     * Generate unique tab ID
     */
    generateTabId() {
        return 'tab_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    }

    /**
     * Get current tab ID
     */
    getTabId() {
        return this.tabId;
    }
}

// Initialize global sync instance
window.AnnaSync = new AnnaSync();

// Setup message history navigation when DOM is ready
document.addEventListener('DOMContentLoaded', function () {
    if (window.AnnaSync) {
        window.AnnaSync.setupMessageHistoryNavigation();
    }
});