/**
 * Anna Sync Module
 * Handles cross-tab synchronization, message history navigation, and autocomplete features
 */

class AnnaSync {
    constructor() {
        this.messageHistory = [];
        this.historyIndex = -1;
        this.currentInput = '';
        this.isNavigating = false;
        this.storageKey = 'pc_message_history';
        this.maxHistorySize = 100;
        this.tabId = null; // Will be generated when needed
        
        // Autocomplete features
        this.autocompleteData = [];
        this.autocompleteCache = {};
        this.autocompleteTimeout = null;
        this.isAutocompleting = false;
        
        // Context suggestions
        this.contextSuggestions = [
            { label: 'Customer', value: '@customer:', icon: '👤' },
            { label: 'Invoice', value: '@invoice:', icon: '📄' },
            { label: 'Project', value: '@project:', icon: '📊' },
            { label: 'Lead', value: '@lead:', icon: '🎯' },
            { label: 'Task', value: '@task:', icon: '✅' },
            { label: 'Proposal', value: '@proposal:', icon: '📋' },
            { label: 'Estimate', value: '@estimate:', icon: '💰' },
            { label: 'Expense', value: '@expense:', icon: '💸' },
            { label: 'Contract', value: '@contract:', icon: '📜' },
            { label: 'Ticket', value: '@ticket:', icon: '🎫' }
        ];
        
        // Load message history from localStorage
        this.loadMessageHistory();
        
        // Set up storage listener for cross-tab sync
        this.setupStorageListener();        
    }

    /**
     * Load message history from localStorage
     */
    loadMessageHistory() {
        try {
            const stored = localStorage.getItem(this.storageKey);
            if (stored) {
                this.messageHistory = JSON.parse(stored);
            }
        } catch (error) {
            this.messageHistory = [];
        }
    }

    /**
     * Save message history to localStorage
     */
    saveMessageHistory() {
        try {
            localStorage.setItem(this.storageKey, JSON.stringify(this.messageHistory));
        } catch (error) {
        }
    }

    /**
     * Add message to history
     */
    addMessageToHistory(message) {
        if (!message || message.trim() === '') return;
        
        const trimmedMessage = message.trim();
        
        // Remove from history if it already exists
        const existingIndex = this.messageHistory.indexOf(trimmedMessage);
        if (existingIndex > -1) {
            this.messageHistory.splice(existingIndex, 1);
        }
        
        // Add to the end
        this.messageHistory.push(trimmedMessage);
        
        // Keep only the last maxHistorySize messages
        if (this.messageHistory.length > this.maxHistorySize) {
            this.messageHistory = this.messageHistory.slice(-this.maxHistorySize);
        }
        
        // Reset navigation
        this.historyIndex = -1;
        this.currentInput = '';
        
        // Save to localStorage
        this.saveMessageHistory();
    }

    /**
     * Navigate through message history
     */
    navigateHistory(direction, inputElement) {
        if (!inputElement || this.messageHistory.length === 0) {
            return;
        }
        
        // Store current input if not already navigating
        if (!this.isNavigating) {
            this.currentInput = inputElement.value;
            this.isNavigating = true;
        }
        
        const oldIndex = this.historyIndex;
        
        if (direction === 'up') {
            // Go to previous message (older)
            if (this.historyIndex < this.messageHistory.length - 1) {
                this.historyIndex++;
            }
        } else if (direction === 'down') {
            // Go to next message (newer)
            if (this.historyIndex > -1) {
                this.historyIndex--;
            }
        }
        
        // Update input value
        if (this.historyIndex === -1) {
            // Back to current input
            inputElement.value = this.currentInput;
            this.isNavigating = false;
        } else {
            // Show message from history
            const historyMessage = this.messageHistory[this.messageHistory.length - 1 - this.historyIndex];
            inputElement.value = historyMessage;
        }
        
        // Move cursor to end
        inputElement.setSelectionRange(inputElement.value.length, inputElement.value.length);
    }

    /**
     * Handle keydown events for history navigation
     */
    handleKeyDown(event) {
        
        if (event.key === 'ArrowUp') {
            event.preventDefault();
            this.navigateHistory('up', event.target);
        } else if (event.key === 'ArrowDown') {
            event.preventDefault();
            this.navigateHistory('down', event.target);
        } else if (event.key === 'Enter' && !event.shiftKey) {
            // Hide suggestions when sending
            this.hideContextSuggestions();
            // Reset navigation state when sending
            this.isNavigating = false;
            this.historyIndex = -1;
            this.currentInput = '';
        } else if (event.key === 'Escape') {
            // Hide suggestions first
            this.hideContextSuggestions();
            // Reset to current input
            if (this.isNavigating) {
                event.target.value = this.currentInput;
                this.isNavigating = false;
                this.historyIndex = -1;
            }
        } else if (event.key === 'Tab') {
            // Handle tab completion
            this.handleTabCompletion(event);
        }
    }

    /**
     * Handle input events
     */
    handleInput(event) {
        // If user starts typing while navigating, exit navigation mode
        if (this.isNavigating) {
            this.isNavigating = false;
            this.historyIndex = -1;
            this.currentInput = event.target.value;
        }
        
        // Handle autocomplete
        this.handleAutocomplete(event);
    }
    
    /**
     * Handle autocomplete for context suggestions
     */
    handleAutocomplete(event) {
        const input = event.target;
        const value = input.value;
        const cursorPosition = input.selectionStart;
        
        // Clear existing timeout
        if (this.autocompleteTimeout) {
            clearTimeout(this.autocompleteTimeout);
        }
        
        // Check if we should show context suggestions
        if (value.includes('@')) {
            this.autocompleteTimeout = setTimeout(() => {
                this.showContextSuggestions(input, value, cursorPosition);
            }, 300);
        } else {
            this.hideContextSuggestions();
        }
    }
    
    /**
     * Show context suggestions dropdown
     */
    showContextSuggestions(input, value, cursorPosition) {
        const atPosition = value.lastIndexOf('@', cursorPosition - 1);
        if (atPosition === -1) return;
        
        const query = value.substring(atPosition + 1, cursorPosition).toLowerCase();
        const matchingSuggestions = this.contextSuggestions.filter(item => 
            item.label.toLowerCase().includes(query) || 
            item.value.toLowerCase().includes(query)
        );
        
        if (matchingSuggestions.length === 0) {
            this.hideContextSuggestions();
            return;
        }
        
        this.renderContextSuggestions(matchingSuggestions, input, atPosition);
    }
    
    /**
     * Render context suggestions dropdown
     */
    renderContextSuggestions(suggestions, input, atPosition) {
        let dropdown = document.getElementById('pc-context-suggestions');
        
        if (!dropdown) {
            dropdown = document.createElement('div');
            dropdown.id = 'pc-context-suggestions';
            dropdown.className = 'pc-context-suggestions-dropdown';
            dropdown.style.cssText = `
                position: absolute;
                background: white;
                border: 1px solid #e2e8f0;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                max-height: 200px;
                overflow-y: auto;
                z-index: 1000;
                font-size: 14px;
                font-family: inherit;
            `;
            document.body.appendChild(dropdown);
        }
        
        // Position dropdown
        const inputRect = input.getBoundingClientRect();
        dropdown.style.left = inputRect.left + 'px';
        dropdown.style.top = (inputRect.bottom + 4) + 'px';
        dropdown.style.width = inputRect.width + 'px';
        
        // Render suggestions
        dropdown.innerHTML = suggestions.map((item, index) => `
            <div class="pc-context-suggestion-item" data-index="${index}" style="
                padding: 8px 12px;
                cursor: pointer;
                display: flex;
                align-items: center;
                gap: 8px;
                transition: background-color 0.2s;
            ">
                <span style="font-size: 16px;">${item.icon}</span>
                <span style="font-weight: 500;">${item.label}</span>
                <span style="color: #64748b; font-size: 12px;">${item.value}</span>
            </div>
        `).join('');
        
        // Add hover effects
        dropdown.querySelectorAll('.pc-context-suggestion-item').forEach(item => {
            item.addEventListener('mouseenter', () => {
                item.style.backgroundColor = '#f1f5f9';
            });
            item.addEventListener('mouseleave', () => {
                item.style.backgroundColor = '';
            });
            item.addEventListener('click', () => {
                this.selectContextSuggestion(suggestions[item.dataset.index], input, atPosition);
            });
        });
        
        dropdown.style.display = 'block';
    }
    
    /**
     * Select a context suggestion
     */
    selectContextSuggestion(suggestion, input, atPosition) {
        const currentValue = input.value;
        const cursorPosition = input.selectionStart;
        
        // Find the end of the current @ mention
        let endPosition = cursorPosition;
        while (endPosition < currentValue.length && currentValue[endPosition] !== ' ') {
            endPosition++;
        }
        
        // Replace the @ mention with the suggestion
        const newValue = currentValue.substring(0, atPosition) + 
                        suggestion.value + ' ' + 
                        currentValue.substring(endPosition);
        
        input.value = newValue;
        
        // Set cursor position after the inserted text
        const newCursorPosition = atPosition + suggestion.value.length + 1;
        input.setSelectionRange(newCursorPosition, newCursorPosition);
        
        // Hide suggestions and focus input
        this.hideContextSuggestions();
        input.focus();
        
        // Trigger change event
        input.dispatchEvent(new Event('input', { bubbles: true }));
    }
    
    /**
     * Hide context suggestions dropdown
     */
    hideContextSuggestions() {
        const dropdown = document.getElementById('pc-context-suggestions');
        if (dropdown) {
            dropdown.style.display = 'none';
        }
    }
    
    /**
     * Handle Tab key for autocomplete
     */
    handleTabCompletion(event) {
        const dropdown = document.getElementById('pc-context-suggestions');
        if (dropdown && dropdown.style.display === 'block') {
            event.preventDefault();
            const firstItem = dropdown.querySelector('.pc-context-suggestion-item');
            if (firstItem) {
                firstItem.click();
            }
        }
    }

    /**
     * Set up message history navigation for an input element
     */
    setupMessageHistoryNavigation(inputSelector = '#pc-widget-input') {
        const inputElement = document.querySelector(inputSelector);
        if (!inputElement) {
            return;
        }

        // Remove existing listeners
        if (this.boundHandleKeyDown) {
            inputElement.removeEventListener('keydown', this.boundHandleKeyDown);
        }
        if (this.boundHandleInput) {
            inputElement.removeEventListener('input', this.boundHandleInput);
        }
        if (this.boundHandleOutsideClick) {
            document.removeEventListener('click', this.boundHandleOutsideClick);
        }
        
        // Bind methods to preserve context
        this.boundHandleKeyDown = this.handleKeyDown.bind(this);
        this.boundHandleInput = this.handleInput.bind(this);
        this.boundHandleOutsideClick = this.handleOutsideClick.bind(this);
        
        // Add event listeners
        inputElement.addEventListener('keydown', this.boundHandleKeyDown);
        inputElement.addEventListener('input', this.boundHandleInput);
        document.addEventListener('click', this.boundHandleOutsideClick);
        
        // Test the event listener
        setTimeout(() => {
            const testEvent = new KeyboardEvent('keydown', { key: 'ArrowUp' });
            inputElement.dispatchEvent(testEvent);
        }, 100);
    }
    
    /**
     * Handle clicks outside the autocomplete dropdown
     */
    handleOutsideClick(event) {
        const dropdown = document.getElementById('pc-context-suggestions');
        if (dropdown && dropdown.style.display === 'block') {
            // Check if the click was inside the dropdown or on the input
            const isInsideDropdown = dropdown.contains(event.target);
            const isOnInput = event.target.matches('#pc-widget-input');
            
            if (!isInsideDropdown && !isOnInput) {
                this.hideContextSuggestions();
            }
        }
    }

    /**
     * Broadcast message to other tabs
     */
    broadcastMessage(conversationId, message, response) {
        const data = {
            type: 'new_message',
            conversation_id: conversationId,
            message: message,
            response: response,
            timestamp: Date.now(),
            tab_id: this.getTabId()
        };
        
        // Add to message history
        this.addMessageToHistory(message);
        
        // Store conversation ID for sync
        localStorage.setItem('pc_current_conversation_id', conversationId);
        
        // Use a unique key with timestamp to ensure localStorage event fires
        const syncKey = 'pc_cross_tab_sync_' + Date.now();
        localStorage.setItem(syncKey, JSON.stringify(data));
        
        // Clean up old sync keys
        this.cleanupOldSyncKeys();
    }

    /**
     * Broadcast conversation switch to other tabs
     */
    broadcastConversationSwitch(conversationId, conversationTitle) {
        const data = {
            type: 'conversation_switch',
            conversation_id: conversationId,
            conversation_title: conversationTitle,
            timestamp: Date.now()
        };
        
        localStorage.setItem('pc_cross_tab_sync', JSON.stringify(data));
    }

    /**
     * Broadcast new conversation to other tabs
     */
    broadcastNewConversation(conversationId, conversationTitle) {
        const data = {
            type: 'new_conversation',
            conversation_id: conversationId,
            conversation_title: conversationTitle,
            timestamp: Date.now()
        };
        
        localStorage.setItem('pc_cross_tab_sync', JSON.stringify(data));
    }

    /**
     * Set up storage listener for cross-tab communication
     */
    setupStorageListener() {
        window.addEventListener('storage', (event) => {
            // Handle cross-tab sync messages
            if (event.key && event.key.startsWith('pc_cross_tab_sync_') && event.newValue) {
                try {
                    const data = JSON.parse(event.newValue);
                    // Don't handle messages from the same tab
                    if (data.tab_id !== this.getTabId()) {
                        this.handleCrossTabMessage(data);
                    }
                } catch (error) {
                    console.error('[Anna Sync] Error parsing cross-tab message:', error);
                }
            }
            
            // Handle conversation ID changes
            if (event.key === 'pc_current_conversation_id' && event.newValue) {
                if (event.newValue !== window.widgetCurrentConversationId) {
                    window.widgetCurrentConversationId = event.newValue;
                    this.loadConversationFromSync(event.newValue);
                }
            }
        });
    }

    /**
     * Handle cross-tab messages
     */
    handleCrossTabMessage(data) {
        switch (data.type) {
            case 'new_message':
                this.syncNewMessage(data);
                break;
            case 'conversation_switch':
                this.syncConversationSwitch(data);
                break;
            case 'new_conversation':
                this.syncNewConversation(data);
                break;
        }
    }

    /**
     * Sync new message across tabs
     */
    syncNewMessage(data) {
        // Update conversation ID if provided
        if (data.conversation_id) {
            window.widgetCurrentConversationId = data.conversation_id;
        }
        
        // Hide welcome screen if visible
        const welcomeScreen = document.getElementById('pc-drawer-welcome');
        if (welcomeScreen && welcomeScreen.style.display !== 'none') {
            welcomeScreen.style.display = 'none';
        }
        
        // Add user message to chat
        if (typeof addWidgetMessage === 'function') {
            addWidgetMessage(data.message, 'user');
            addWidgetMessage(data.response, 'ai');
        }
        
        // Add to message history
        this.addMessageToHistory(data.message);
    }

    /**
     * Sync conversation switch across tabs
     */
    syncConversationSwitch(data) {
        // Update current conversation ID
        window.widgetCurrentConversationId = data.conversation_id;
        
        // Trigger conversation load if function exists
        if (typeof loadWidgetConversation === 'function') {
            loadWidgetConversation(data.conversation_id);
        }
    }

    /**
     * Sync new conversation across tabs
     */
    syncNewConversation(data) {
        // Update current conversation ID
        window.widgetCurrentConversationId = data.conversation_id;
        
        // Clear messages if function exists
        if (typeof clearWidgetMessages === 'function') {
            clearWidgetMessages();
        }
        
        // Update conversation list if function exists
        if (typeof updateWidgetConversationList === 'function') {
            updateWidgetConversationList();
        }
    }

    /**
     * Clear message history
     */
    clearHistory() {
        this.messageHistory = [];
        this.historyIndex = -1;
        this.currentInput = '';
        this.isNavigating = false;
        this.saveMessageHistory();
    }

    /**
     * Get message history
     */
    getHistory() {
        return this.messageHistory;
    }

    /**
     * Get current navigation state
     */
    getNavigationState() {
        return {
            isNavigating: this.isNavigating,
            historyIndex: this.historyIndex,
            currentInput: this.currentInput,
            totalMessages: this.messageHistory.length
        };
    }
    
    /**
     * Get unique tab ID
     */
    getTabId() {
        if (!this.tabId) {
            this.tabId = 'tab_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
        }
        return this.tabId;
    }
    
    /**
     * Clean up old sync keys from localStorage
     */
    cleanupOldSyncKeys() {
        const now = Date.now();
        const maxAge = 5000; // 5 seconds
        
        for (let i = localStorage.length - 1; i >= 0; i--) {
            const key = localStorage.key(i);
            if (key && key.startsWith('pc_cross_tab_sync_')) {
                const timestamp = parseInt(key.split('_').pop());
                if (now - timestamp > maxAge) {
                    localStorage.removeItem(key);
                }
            }
        }
    }
    
    /**
     * Load conversation from sync
     */
    loadConversationFromSync(conversationId) {
        if (window.perfexConversationManager && window.perfexConversationManager.loadConversation) {
            window.perfexConversationManager.loadConversation(conversationId).then(() => {
                // Reload conversation history in widget
                if (window.perfexConversationManager.conversationHistory) {
                    const messagesContainer = document.getElementById('pc-widget-messages');
                    if (messagesContainer) {
                        messagesContainer.innerHTML = '';
                        window.perfexConversationManager.conversationHistory.forEach(msg => {
                            if (msg.role === 'user') {
                                addWidgetMessage(msg.content, 'user');
                            } else if (msg.role === 'assistant' || msg.role === 'ai') {
                                addWidgetMessage(msg.content, 'ai');
                            }
                        });
                        
                        // Hide welcome screen
                        const welcomeScreen = document.getElementById('pc-drawer-welcome');
                        if (welcomeScreen) {
                            welcomeScreen.style.display = 'none';
                        }
                    }
                }
            });
        }
    }
}

// Export the class
window.AnnaSync = AnnaSync;