// modules/anna/assets/js/conversation-manager.js
// Conversation Management System for Anna

class AnnaConversationManager {
    constructor() {
        this.storageKey = 'anna_conversation_id';
        this.historyKey = 'anna_conversation_history';
        this.currentConversationId = null;
        this.conversationHistory = [];
        this.isInitialized = false;

        // Initialize on construction
        this.initialize();
    }

    /**
     * Initialize the conversation manager
     */
    async initialize() {
        if (this.isInitialized) {
            return;
        }

        // Try to get existing conversation ID from storage
        this.currentConversationId = this.getStoredConversationId();

        if (this.currentConversationId) {
            // Load conversation history
            await this.loadConversationHistory();
        } else {
            // Create new conversation
            const newId = await this.createNewConversation();
            if (!newId) {
                // Fallback: generate a temporary ID that will be replaced by server
                this.currentConversationId = null; // Let the server create it
            }
        }

        this.isInitialized = true;

        // Set global variable for backwards compatibility
        window.widgetCurrentConversationId = this.currentConversationId;
        window.currentConversationId = this.currentConversationId;
    }

    /**
     * Get conversation ID from localStorage
     */
    getStoredConversationId() {
        try {
            return localStorage.getItem(this.storageKey);
        } catch (error) {
            return null;
        }
    }

    /**
     * Store conversation ID in localStorage
     */
    storeConversationId(conversationId) {
        try {
            localStorage.setItem(this.storageKey, conversationId);
            this.currentConversationId = conversationId;

            // Update global variables
            window.widgetCurrentConversationId = conversationId;
            window.currentConversationId = conversationId;
        } catch (error) {
            // Silently handle error
        }
    }

    /**
     * Create a new conversation
     */
    async createNewConversation() {
        try {
            const url = admin_url + 'anna/new_conversation';

            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: '' // Send empty body, let server handle everything
            });

            if (!response.ok) {
                return null;
            }

            const data = await response.json();

            if (data.success && data.conversation_id) {
                this.storeConversationId(data.conversation_id);
                this.conversationHistory = [];
                this.storeConversationHistory([]);

                return data.conversation_id;
            } else {
                return null;
            }
        } catch (error) {
            return null;
        }
    }

    /**
     * Load conversation history from server
     */
    async loadConversationHistory() {
        if (!this.currentConversationId) return [];

        try {
            const response = await fetch(admin_url + 'anna/get_conversation/' + this.currentConversationId, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                }
            });

            const data = await response.json();

            if (data.success && data.conversation && data.conversation.messages) {
                this.conversationHistory = data.conversation.messages;
                this.storeConversationHistory(this.conversationHistory);

                return this.conversationHistory;
            } else {
                // If conversation doesn't exist, create a new one
                if (data.error) {
                    await this.createNewConversation();
                }
                return [];
            }
        } catch (error) {
            return [];
        }
    }

    /**
     * Store conversation history in localStorage for offline access
     */
    storeConversationHistory(history) {
        try {
            localStorage.setItem(this.historyKey, JSON.stringify(history));
        } catch (error) {
            console.error('[ConversationManager] Error storing conversation history:', error);
        }
    }

    /**
     * Get stored conversation history from localStorage
     */
    getStoredConversationHistory() {
        try {
            const stored = localStorage.getItem(this.historyKey);
            return stored ? JSON.parse(stored) : [];
        } catch (error) {
            console.error('[ConversationManager] Error reading conversation history:', error);
            return [];
        }
    }

    /**
     * Add a message to the conversation history
     */
    addMessageToHistory(message, role = 'user') {
        const messageObj = {
            id: Date.now(),
            role: role,
            content: message,
            created_at: new Date().toISOString(),
            conversation_id: this.currentConversationId
        };

        this.conversationHistory.push(messageObj);
        this.storeConversationHistory(this.conversationHistory);

        return messageObj;
    }

    /**
     * Get current conversation ID
     */
    getCurrentConversationId() {
        return this.currentConversationId || '';
    }

    /**
     * Get conversation history
     */
    getConversationHistory() {
        return this.conversationHistory;
    }

    /**
     * Force create a new conversation (useful for "New Chat" button)
     */
    async startNewConversation() {
        this.clearStoredData();
        await this.createNewConversation();
        return this.currentConversationId;
    }

    /**
     * Clear stored conversation data
     */
    clearStoredData() {
        try {
            localStorage.removeItem(this.storageKey);
            localStorage.removeItem(this.historyKey);
            this.currentConversationId = null;
            this.conversationHistory = [];

            // Clear global variables
            window.widgetCurrentConversationId = null;
            window.currentConversationId = null;

        } catch (error) {
        }
    }

    /**
     * Load a specific conversation by ID
     */
    async loadConversation(conversationId) {
        if (!conversationId) return false;

        try {
            // Store the new conversation ID
            this.storeConversationId(conversationId);

            // Load its history
            await this.loadConversationHistory();

            return true;
        } catch (error) {
            return false;
        }
    }

    /**
     * Get CSRF form data for requests
     */
    getCsrfFormData() {
        // Try different ways to get CSRF token
        let csrfTokenName = window.csrf_token_name;
        let csrfToken = null;
        let csrfValue = '';

        // Try to find CSRF token in various ways
        if (csrfTokenName) {
            csrfToken = document.querySelector('input[name="' + csrfTokenName + '"]');
        }

        if (!csrfToken) {
            // Fallback: try common CSRF token names
            const commonNames = ['csrf_token_name', 'csrf_test_name', '_token'];
            for (const name of commonNames) {
                csrfToken = document.querySelector('input[name="' + name + '"]');
                if (csrfToken) {
                    csrfTokenName = name;
                    break;
                }
            }
        }

        if (csrfToken) {
            csrfValue = csrfToken.value;
        }

        const formData = new URLSearchParams();
        if (csrfTokenName && csrfValue) {
            formData.append(csrfTokenName, csrfValue);
        }

        return formData.toString();
    }

    /**
     * Render conversation history in a container
     */
    renderHistoryInContainer(containerId) {
        const container = document.getElementById(containerId);
        if (!container) {
            console.error('[ConversationManager] Container not found:', containerId);
            return;
        }

        // Clear existing content
        container.innerHTML = '';

        if (this.conversationHistory.length === 0) {
            container.innerHTML = '<div class="no-messages">Start a conversation by sending a message!</div>';
            return;
        }

        // Render messages
        this.conversationHistory.forEach(message => {
            const messageDiv = document.createElement('div');
            messageDiv.className = `message ${message.role}`;

            const timeStr = new Date(message.created_at).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });

            messageDiv.innerHTML = `
                <div class="message-content">
                    <div class="message-text">${this.escapeHtml(message.content)}</div>
                    <div class="message-time">${timeStr}</div>
                </div>
            `;

            container.appendChild(messageDiv);
        });

        // Scroll to bottom
        container.scrollTop = container.scrollHeight;
    }

    /**
     * Escape HTML to prevent XSS
     */
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    /**
     * Get singleton instance
     */
    static getInstance() {
        if (!window.annaConversationManager) {
            window.annaConversationManager = new AnnaConversationManager();
        }
        return window.annaConversationManager;
    }
}

// Export for use
window.AnnaConversationManager = AnnaConversationManager;

// Auto-initialize when DOM is ready
function initializeConversationManager() {
    try {
        const instance = AnnaConversationManager.getInstance();
        
        // Trigger event to notify other scripts
        window.dispatchEvent(new CustomEvent('anna:conversation-manager-ready', {
            detail: { instance }
        }));

        return instance;
    } catch (error) {
        return null;
    }
}

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeConversationManager);
} else {
    // DOM is already ready
    initializeConversationManager();
}