<?php

// modules/anna/controllers/Anna.php
defined('BASEPATH') or exit('No direct script access allowed');

class Anna extends AdminController
{
    private $error_handler;
    private $debug;
    
    public function __construct()
    {
        parent::__construct();
        $this->load->model('anna_model');
        
        // Initialize error handling and debugging
        $this->load->library('anna_error_handler');
        $this->load->library('anna_debug');
        $this->error_handler = $this->anna_error_handler;
        $this->debug = $this->anna_debug;
        
        // Start debug session
        $this->debug->startSession('anna_controller');
    }

    public function index()
    {
        if (!has_permission('anna', '', 'view')) {
            access_denied('anna');
        }

        $data['title'] = _l('anna');
        $this->load->view('index', $data);
    }

    public function chat($conversation_id = null)
    {
        if (!has_permission('anna', '', 'view')) {
            access_denied('anna');
        }

        if ($this->input->post()) {
            $message = $this->input->post('message');
            $post_conversation_id = $this->input->post('conversation_id');

            try {
                $this->debug->startBenchmark('process_chat');
                $response = $this->anna_model->process_chat($message, $post_conversation_id);
                $current_conversation_id = $this->anna_model->getCurrentConversationId();
                $this->debug->endBenchmark('process_chat');
                
                echo json_encode([
                    'success' => true, 
                    'response' => $response,
                    'conversation_id' => $current_conversation_id,
                ]);
            } catch (Exception $e) {
                $error_response = $this->error_handler->handleApiError(
                    $e->getMessage(), 
                    500, 
                    ['action' => 'chat', 'conversation_id' => $post_conversation_id]
                );
                echo json_encode($error_response);
            }
            return;
        }

        $data['title'] = _l('anna_chat');
        $data['conversation_id'] = $conversation_id;
        $data['conversations'] = $this->anna_model->getConversationList();
        
        if ($conversation_id) {
            $data['current_conversation'] = $this->anna_model->getFullConversation($conversation_id);
        }
        
        $this->load->view('chat_with_sidebar', $data);
    }

    public function settings()
    {
        if (!has_permission('anna', '', 'edit')) {
            access_denied('anna');
        }

        if ($this->input->post()) {
            $settings = $this->input->post();
            
            // Handle regular settings
            foreach ($settings as $key => $value) {
                if (strpos($key, 'tool_') !== 0) {
                    update_option('anna_' . $key, $value);
                }
            }
            
            // Handle tool enable/disable settings
            $all_tools = $this->anna_model->getToolsWithStatus();
            foreach ($all_tools as $tool_name => $tool_data) {
                $setting_name = 'anna_tool_' . $tool_name . '_enabled';
                $is_enabled = isset($settings['tool_' . $tool_name . '_enabled']) ? '1' : '0';
                update_option($setting_name, $is_enabled);
            }
            
            set_alert('success', _l('settings_updated'));
            redirect(admin_url('anna/settings'));
        }

        $data['title'] = _l('anna_settings');
        $data['tools'] = $this->anna_model->getToolsWithStatus();
        $this->load->view('settings', $data);
    }

    public function get_plans()
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        try {
            $this->load->config('anna/conf');
            $plans_api_url = $this->config->item('anna_plans_api_url');
            
            if (empty($plans_api_url)) {
                echo json_encode(['error' => 'Plans API URL not configured']);
                return;
            }

            $token = '';

            if (file_exists(APP_MODULES_PATH . '/' . ANNA_MODULE_NAME . '/config/token.php')) {
                $token    = file_get_contents(APP_MODULES_PATH . '/' . ANNA_MODULE_NAME . '/config/token.php');
            }

            // Make request to plans API via cURL
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $plans_api_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                'site_url' => site_url(),
                'token' => $token
            ]));
            curl_setopt($ch, CURLOPT_TIMEOUT, 120); // 2 minutes timeout for plans API
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // 10 seconds to establish connection
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json'
            ]);
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_error($ch)) {
                echo json_encode(['error' => 'Failed to connect to plans API: ' . curl_error($ch)]);
                curl_close($ch);
                return;
            }
            
            curl_close($ch);
            
            if ($http_code !== 200) {
                echo json_encode(['error' => 'Plans API returned status: ' . $http_code, 'response' => json_decode($response, true)]);
                return;
            }
            
            $data = json_decode($response, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                echo json_encode(['error' => 'Invalid JSON response from plans API']);
                return;
            }
            
            // Return the plans data
            header('Content-Type: application/json');
            echo json_encode($data);
            
        } catch (Exception $e) {
            $error_response = $this->error_handler->handleApiError(
                $e->getMessage(),
                500,
                ['action' => 'get_plans']
            );
            echo json_encode($error_response);
        }
    }

    public function process_chat()
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        $message = $this->input->post('message');
        $conversation_id = $this->input->post('conversation_id');
        $context = $this->input->post('context') ?: 'general';
        $key = $this->input->post('key');
        $auth_key = $this->input->post('auth_key');

        if (empty($message)) {
            echo json_encode(['error' => 'Message is required']);
            return;
        }

        // If no conversation ID provided, create a new one
        if (empty($conversation_id)) {
            log_message('debug', '[Anna] No conversation ID provided, creating new conversation');
            try {
                $conversation_id = $this->anna_model->createNewConversation();
                if (!$conversation_id) {
                    log_message('error', '[Anna] Failed to create new conversation');
                    echo json_encode(['error' => 'Failed to create new conversation']);
                    return;
                }
                log_message('debug', '[Anna] Created new conversation: ' . $conversation_id);
            } catch (Exception $e) {
                log_message('error', '[Anna] Exception creating conversation: ' . $e->getMessage());
                echo json_encode(['error' => 'Failed to create conversation: ' . $e->getMessage()]);
                return;
            }
        } else {
            log_message('debug', '[Anna] Using existing conversation ID: ' . $conversation_id);
        }

        try {
            $this->load->config('anna/conf');
            $api_url = $this->config->item('anna_api_url');
            
            if (empty($api_url)) {
                echo json_encode(['error' => 'Chat API URL not configured']);
                return;
            }

            $token = '';

            if (file_exists(APP_MODULES_PATH . '/' . ANNA_MODULE_NAME . '/config/token.php')) {
                $token    = file_get_contents(APP_MODULES_PATH . '/' . ANNA_MODULE_NAME . '/config/token.php');
            }

            $staff = get_staff();
            $staff_data = [
                'full_name' => $staff->full_name,
                'email' => $staff->email,
                'phone' => $staff->phonenumber,
                'staffid' => $staff->staffid
            ];

            // Prepare request data for Python API
            $request_data = [
                'message' => $message,
                'thread_id' => (string) $conversation_id,
                'context' => $context,
                'site_url' => site_url(),
                'token' => $token,
                'key' => $key,
                'auth_key' => $auth_key,
                'staff' => $staff_data,
                'user_timezone' => date_default_timezone_get(),
                'datetime' => date('Y-m-d H:i:s')
            ];

            // Make request to Python chat API
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $api_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($request_data));
            curl_setopt($ch, CURLOPT_TIMEOUT, 300); // 5 minutes timeout
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // 10 seconds to establish connection
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json'
            ]);
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curl_error = curl_error($ch);
            
            curl_close($ch);
            
            // If API server is not available, fall back to model response
            if ($curl_error || $http_code !== 200) {
                log_message('debug', '[Anna] API server unavailable, using model fallback. Error: ' . $curl_error . ', HTTP Code: ' . $http_code);
                
                // Use model fallback
                $fallback_response = $this->anna_model->process_chat($message, $conversation_id);
                $current_conversation_id = $this->anna_model->getCurrentConversationId() ?: $conversation_id;
                
                echo json_encode([
                    'success' => true,
                    'response' => $curl_error,
                    'conversation_id' => $current_conversation_id,
                    'session_message' => 'Using offline mode - API server not available'
                ]);
                return;
            }
            
            $api_response = json_decode($response, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                log_message('debug', '[Anna] Invalid JSON from API, using model fallback');
                
                // Use model fallback for invalid JSON
                $fallback_response = $this->anna_model->process_chat($message, $conversation_id);
                $current_conversation_id = $this->anna_model->getCurrentConversationId() ?: $conversation_id;
                
                echo json_encode([
                    'success' => true,
                    'response' => $fallback_response,
                    'conversation_id' => $current_conversation_id,
                    'session_message' => 'Using offline mode - Invalid API response'
                ]);
                return;
            }

            // Store conversation in database using existing model methods
            if (isset($api_response['response'])) {
                $chat_response = $api_response['response'];
                
                // Extract session_message if present
                $session_message = isset($api_response['session_message']) ? $api_response['session_message'] : '';
                
                // Log the conversation using the existing model method
                $this->anna_model->log_chat_interaction($message, $chat_response, '', $conversation_id);
                
                // Use the conversation ID we have (either provided or newly created)
                $current_conversation_id = $conversation_id;
                
                // Prepare response with session_message field
                $response_data = [
                    'success' => true,
                    'response' => $chat_response,
                    'conversation_id' => $current_conversation_id
                ];
                
                // Add session_message if it's not empty
                if (!empty($session_message)) {
                    $response_data['session_message'] = $session_message;
                }
                
                echo json_encode($response_data);
            } else {
                echo json_encode(['error' => 'Invalid response from chat API']);
            }
            
        } catch (Exception $e) {
            $error_response = $this->error_handler->handleApiError(
                $e->getMessage(),
                500,
                ['action' => 'process_chat']
            );
            echo json_encode($error_response);
        }
    }

    public function new_conversation()
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        try {
            // Force create new conversation by passing null
            $conversation_id = $this->anna_model->createNewConversation();
            echo json_encode(['success' => true, 'conversation_id' => $conversation_id]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function get_conversations()
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        try {
            $conversations = $this->anna_model->getConversationList();
            echo json_encode(['success' => true, 'conversations' => $conversations]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function get_conversation($conversation_id)
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        try {
            $conversation = $this->anna_model->getFullConversation($conversation_id);
            echo json_encode(['success' => true, 'conversation' => $conversation]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function delete_conversation($conversation_id)
    {
        if (!has_permission('anna', '', 'edit')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        try {
            // Add some debugging info
            log_message('debug', "Attempting to delete conversation ID: $conversation_id by staff ID: " . get_staff_user_id() . " (admin: " . (is_admin() ? 'yes' : 'no') . ")");
            
            $result = $this->anna_model->deleteConversation($conversation_id);
            if ($result) {
                log_message('debug', "Successfully deleted conversation ID: $conversation_id");
                echo json_encode(['success' => true, 'message' => 'Conversation deleted successfully']);
            } else {
                log_message('debug', "Failed to delete conversation ID: $conversation_id - conversation not found or access denied");
                echo json_encode(['success' => false, 'error' => 'Conversation not found or access denied']);
            }
        } catch (Exception $e) {
            log_message('error', 'Error deleting conversation: ' . $e->getMessage());
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function rename_conversation($conversation_id)
    {
        if (!has_permission('anna', '', 'edit')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        $new_title = $this->input->post('title');
        if (!$new_title) {
            echo json_encode(['success' => false, 'error' => 'Title is required']);
            return;
        }

        try {
            $this->anna_model->renameConversation($conversation_id, $new_title);
            echo json_encode(['success' => true]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function debug_conversations()
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        echo "<h3>Conversations Debug</h3>";
        echo "Current Staff ID: " . get_staff_user_id() . "<br>";
        echo "Is Admin: " . (is_admin() ? 'YES' : 'NO') . "<br>";
        echo "DB Prefix: " . db_prefix() . "<br><br>";
        
        // Check if tables exist
        $table_name = db_prefix() . 'anna_conversations';
        echo "Table name: $table_name<br>";
        echo "Table exists: " . ($this->db->table_exists($table_name) ? 'YES' : 'NO') . "<br><br>";
        
        if (!$this->db->table_exists($table_name)) {
            echo "<strong>ERROR: Table doesn't exist! Creating tables...</strong><br>";
            $this->anna_model->ensureConversationTablesExist();
            echo "Tables created. <a href=''>Refresh this page</a><br>";
            return;
        }
        
        // Get all conversations
        $this->db->select('*');
        $this->db->from($table_name);
        $this->db->order_by('created_at', 'DESC');
        $query = $this->db->get();
        
        if ($this->db->error()['code'] != 0) {
            echo "<strong>Database Error:</strong> " . $this->db->error()['message'] . "<br>";
            return;
        }
        
        $conversations = $query->result_array();
        
        echo "<h4>All Conversations (" . count($conversations) . " found):</h4>";
        if (empty($conversations)) {
            echo "No conversations found.<br>";
        } else {
            echo "<table border='1' style='border-collapse: collapse; padding: 5px;'>";
            echo "<tr><th>ID</th><th>Staff ID</th><th>Title</th><th>Status</th><th>Created</th><th>Can Delete?</th></tr>";
            
            foreach ($conversations as $conv) {
                $can_delete = is_admin() || $conv['staff_id'] == get_staff_user_id();
                echo "<tr>";
                echo "<td>{$conv['id']}</td>";
                echo "<td>{$conv['staff_id']}</td>";
                echo "<td>" . htmlspecialchars($conv['title']) . "</td>";
                echo "<td>{$conv['status']}</td>";
                echo "<td>{$conv['created_at']}</td>";
                echo "<td>" . ($can_delete ? 'YES' : 'NO') . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        }
        
        echo "<br><h4>Test Delete Function:</h4>";
        echo "To test delete, add <code>?test_delete=CONVERSATION_ID</code> to this URL";
        
        if ($this->input->get('test_delete')) {
            $test_id = $this->input->get('test_delete');
            echo "<br><strong>Testing delete for conversation $test_id:</strong><br>";
            $result = $this->anna_model->deleteConversation($test_id);
            echo "Result: " . ($result ? 'SUCCESS' : 'FAILED') . "<br>";
        }
    }

    public function update_settings()
    {
        if (!has_permission('anna', '', 'edit')) {
            access_denied('anna');
        }

        if ($this->input->post()) {
            $settings = $this->input->post();
            
            // Handle settings update
            foreach ($settings as $key => $value) {
                if (strpos($key, 'anna_') === 0) {
                    update_option($key, $value);
                }
            }
            
            set_alert('success', _l('settings_updated'));
        }
        
        redirect(admin_url('settings?group=anna'));
    }

    /**
     * Error monitoring dashboard
     */
    public function error_monitoring()
    {
        if (!has_permission('anna', '', 'edit')) {
            access_denied('anna');
        }

        $data['title'] = 'Error Monitoring & Debugging';
        $this->load->view('error_monitoring', $data);
    }

    /**
     * Get error statistics
     */
    public function get_error_stats()
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['success' => false, 'error' => 'Access denied']);
            return;
        }

        try {
            $stats = $this->error_handler->getErrorStats(24);
            echo json_encode(['success' => true, 'stats' => $stats]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    /**
     * Get recent errors
     */
    public function get_recent_errors()
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['success' => false, 'error' => 'Access denied']);
            return;
        }

        try {
            $this->db->select('error_id, timestamp, level, category, message, staff_id');
            $this->db->from(db_prefix() . 'anna_error_log');
            $this->db->order_by('timestamp', 'DESC');
            $this->db->limit(20);
            
            $errors = $this->db->get()->result_array();
            echo json_encode(['success' => true, 'errors' => $errors]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    /**
     * Get error details
     */
    public function get_error_details($error_id)
    {
        if (!has_permission('anna', '', 'view')) {
            echo json_encode(['success' => false, 'error' => 'Access denied']);
            return;
        }

        try {
            $this->db->where('error_id', $error_id);
            $error = $this->db->get(db_prefix() . 'anna_error_log')->row_array();
            
            if ($error) {
                echo json_encode(['success' => true, 'error' => $error]);
            } else {
                echo json_encode(['success' => false, 'error' => 'Error not found']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    /**
     * Generate debug report
     */
    public function generate_debug_report()
    {
        if (!has_permission('anna', '', 'edit')) {
            access_denied('anna');
        }

        try {
            $format = $this->input->get('format') ?: 'html';
            $report = $this->debug->generateDebugReport($format);
            
            $filename = 'anna_debug_report_' . date('Y-m-d_H-i-s');
            
            switch ($format) {
                case 'html':
                    header('Content-Type: text/html');
                    header('Content-Disposition: attachment; filename="' . $filename . '.html"');
                    echo $report;
                    break;
                case 'json':
                    header('Content-Type: application/json');
                    header('Content-Disposition: attachment; filename="' . $filename . '.json"');
                    echo $report;
                    break;
                case 'text':
                    header('Content-Type: text/plain');
                    header('Content-Disposition: attachment; filename="' . $filename . '.txt"');
                    echo $report;
                    break;
                default:
                    echo $report;
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    /**
     * Cleanup old logs
     */
    public function cleanup_logs()
    {
        if (!has_permission('anna', '', 'edit')) {
            echo json_encode(['success' => false, 'error' => 'Access denied']);
            return;
        }

        try {
            $days = get_option('anna_log_retention_days', 30);
            $deleted_count = $this->error_handler->cleanupOldLogs($days);
            
            // Also cleanup debug sessions
            $this->debug->cleanupOldSessions($days);
            
            echo json_encode(['success' => true, 'deleted_count' => $deleted_count]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    /**
     * Test error handling
     */
    public function test_error_handling()
    {
        if (!has_permission('anna', '', 'edit')) {
            echo json_encode(['success' => false, 'error' => 'Access denied']);
            return;
        }

        try {
            // Test different error levels
            $this->error_handler->logError('Debug test message', 
                Anna_error_handler::LEVEL_DEBUG, 
                Anna_error_handler::CATEGORY_SYSTEM, 
                ['test' => true]
            );
            
            $this->error_handler->logError('Info test message', 
                Anna_error_handler::LEVEL_INFO, 
                Anna_error_handler::CATEGORY_SYSTEM, 
                ['test' => true]
            );
            
            $this->error_handler->logError('Warning test message', 
                Anna_error_handler::LEVEL_WARNING, 
                Anna_error_handler::CATEGORY_SYSTEM, 
                ['test' => true]
            );
            
            $this->error_handler->logError('Error test message', 
                Anna_error_handler::LEVEL_ERROR, 
                Anna_error_handler::CATEGORY_SYSTEM, 
                ['test' => true]
            );
            
            echo json_encode(['success' => true, 'message' => 'Test errors logged successfully']);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    /**
     * Update error monitoring settings
     */
    public function update_error_settings()
    {
        if (!has_permission('anna', '', 'edit')) {
            access_denied('anna');
        }

        if ($this->input->post()) {
            $settings = $this->input->post();
            
            $allowed_settings = [
                'anna_debug_enabled',
                'anna_performance_monitoring', 
                'anna_save_debug_sessions',
                'anna_error_reporting_email',
                'anna_log_retention_days'
            ];
            
            foreach ($allowed_settings as $setting) {
                $value = isset($settings[$setting]) ? $settings[$setting] : '0';
                update_option($setting, $value);
            }
            
            set_alert('success', 'Error monitoring settings updated successfully');
            redirect(admin_url('anna/error_monitoring'));
        }
        
        redirect(admin_url('anna/error_monitoring'));
    }
}
