<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Anna Error Handling Helper Functions
 * 
 * Helper functions to easily integrate error handling throughout the application
 */

if (!function_exists('anna_log_error')) {
    /**
     * Log an error using the Anna error handler
     * 
     * @param string $message The error message
     * @param string $level Error level (DEBUG, INFO, WARNING, ERROR, CRITICAL)
     * @param string $category Error category (API, DATABASE, AUTH, etc.)
     * @param array $context Additional context data
     * @return string Error ID
     */
    function anna_log_error($message, $level = 'ERROR', $category = 'SYSTEM', $context = [])
    {
        $CI = &get_instance();
        
        // Load error handler if not loaded
        if (!isset($CI->anna_error_handler)) {
            $CI->load->library('anna_error_handler');
        }
        
        return $CI->anna_error_handler->logError($message, $level, $category, $context);
    }
}

if (!function_exists('anna_handle_api_error')) {
    /**
     * Handle API error with standardized response
     * 
     * @param string $error Error message
     * @param int $http_code HTTP status code
     * @param array $context Additional context
     * @return array Standardized error response
     */
    function anna_handle_api_error($error, $http_code = 500, $context = [])
    {
        $CI = &get_instance();
        
        if (!isset($CI->anna_error_handler)) {
            $CI->load->library('anna_error_handler');
        }
        
        return $CI->anna_error_handler->handleApiError($error, $http_code, $context);
    }
}

if (!function_exists('anna_handle_db_error')) {
    /**
     * Handle database error
     * 
     * @param string $error Error message
     * @param string $query SQL query that caused the error
     * @param array $context Additional context
     * @return string Error ID
     */
    function anna_handle_db_error($error, $query = null, $context = [])
    {
        $CI = &get_instance();
        
        if (!isset($CI->anna_error_handler)) {
            $CI->load->library('anna_error_handler');
        }
        
        return $CI->anna_error_handler->handleDatabaseError($error, $query, $context);
    }
}

if (!function_exists('anna_handle_validation_error')) {
    /**
     * Handle validation error
     * 
     * @param array $errors Validation errors
     * @param array $context Additional context
     * @return array Standardized validation error response
     */
    function anna_handle_validation_error($errors, $context = [])
    {
        $CI = &get_instance();
        
        if (!isset($CI->anna_error_handler)) {
            $CI->load->library('anna_error_handler');
        }
        
        return $CI->anna_error_handler->handleValidationError($errors, $context);
    }
}

if (!function_exists('anna_debug_event')) {
    /**
     * Log a debug event
     * 
     * @param string $event_name Event name
     * @param array $data Event data
     * @param string $session_name Debug session name
     */
    function anna_debug_event($event_name, $data = [], $session_name = 'default')
    {
        $CI = &get_instance();
        
        if (!isset($CI->anna_debug)) {
            $CI->load->library('anna_debug');
        }
        
        $CI->anna_debug->logDebugEvent($session_name, $event_name, $data);
    }
}

if (!function_exists('anna_start_benchmark')) {
    /**
     * Start a performance benchmark
     * 
     * @param string $name Benchmark name
     */
    function anna_start_benchmark($name)
    {
        $CI = &get_instance();
        
        if (!isset($CI->anna_debug)) {
            $CI->load->library('anna_debug');
        }
        
        $CI->anna_debug->startBenchmark($name);
    }
}

if (!function_exists('anna_end_benchmark')) {
    /**
     * End a performance benchmark
     * 
     * @param string $name Benchmark name
     * @return array Benchmark results
     */
    function anna_end_benchmark($name)
    {
        $CI = &get_instance();
        
        if (!isset($CI->anna_debug)) {
            $CI->load->library('anna_debug');
        }
        
        return $CI->anna_debug->endBenchmark($name);
    }
}

if (!function_exists('anna_is_debug_enabled')) {
    /**
     * Check if debug mode is enabled
     * 
     * @return bool
     */
    function anna_is_debug_enabled()
    {
        return get_option('anna_debug_enabled', '0') === '1';
    }
}

if (!function_exists('anna_is_performance_monitoring_enabled')) {
    /**
     * Check if performance monitoring is enabled
     * 
     * @return bool
     */
    function anna_is_performance_monitoring_enabled()
    {
        return get_option('anna_performance_monitoring', '0') === '1';
    }
}

if (!function_exists('anna_safe_execute')) {
    /**
     * Execute a function with error handling
     * 
     * @param callable $callback Function to execute
     * @param array $fallback_response Fallback response on error
     * @param string $error_context Error context for logging
     * @return mixed Result of callback or fallback response
     */
    function anna_safe_execute($callback, $fallback_response = [], $error_context = 'unknown')
    {
        try {
            return $callback();
        } catch (Exception $e) {
            anna_log_error(
                'Safe execution failed: ' . $e->getMessage(),
                'ERROR',
                'SYSTEM',
                [
                    'context' => $error_context,
                    'exception' => get_class($e),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]
            );
            
            return $fallback_response;
        }
    }
}

if (!function_exists('anna_monitor_query')) {
    /**
     * Monitor database query performance
     * 
     * @param string $query SQL query
     * @param float $execution_time Query execution time
     * @param int $result_count Number of results
     */
    function anna_monitor_query($query, $execution_time = null, $result_count = null)
    {
        $CI = &get_instance();
        
        if (!isset($CI->anna_debug)) {
            $CI->load->library('anna_debug');
        }
        
        $CI->anna_debug->monitorQuery($query, $execution_time, $result_count);
    }
}

if (!function_exists('anna_memory_snapshot')) {
    /**
     * Take a memory usage snapshot
     * 
     * @param string $label Snapshot label
     */
    function anna_memory_snapshot($label = null)
    {
        $CI = &get_instance();
        
        if (!isset($CI->anna_debug)) {
            $CI->load->library('anna_debug');
        }
        
        $CI->anna_debug->takeMemorySnapshot($label);
    }
}

if (!function_exists('anna_format_error_response')) {
    /**
     * Format a standardized error response
     * 
     * @param string $error Error message
     * @param string $error_id Error ID
     * @param int $http_code HTTP status code
     * @param array $additional_data Additional response data
     * @return array Formatted error response
     */
    function anna_format_error_response($error, $error_id = null, $http_code = 500, $additional_data = [])
    {
        $response = [
            'success' => false,
            'error' => $error,
            'timestamp' => date('c'),
            'http_code' => $http_code
        ];
        
        if ($error_id) {
            $response['error_id'] = $error_id;
        }
        
        if (anna_is_debug_enabled()) {
            $response['debug_info'] = array_merge([
                'memory_usage' => memory_get_usage(true),
                'peak_memory' => memory_get_peak_usage(true),
                'execution_time' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']
            ], $additional_data);
        }
        
        return $response;
    }
}

if (!function_exists('anna_format_success_response')) {
    /**
     * Format a standardized success response
     * 
     * @param mixed $data Response data
     * @param string $message Success message
     * @param array $meta Additional metadata
     * @return array Formatted success response
     */
    function anna_format_success_response($data, $message = 'Success', $meta = [])
    {
        $response = [
            'success' => true,
            'message' => $message,
            'data' => $data,
            'timestamp' => date('c')
        ];
        
        if (!empty($meta)) {
            $response['meta'] = $meta;
        }
        
        if (anna_is_debug_enabled()) {
            $response['debug_info'] = [
                'memory_usage' => memory_get_usage(true),
                'peak_memory' => memory_get_peak_usage(true),
                'execution_time' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']
            ];
        }
        
        return $response;
    }
}

if (!function_exists('anna_with_error_handling')) {
    /**
     * Wrap a function with comprehensive error handling
     * 
     * @param callable $callback Function to execute
     * @param string $operation_name Operation name for logging
     * @param array $default_response Default response on error
     * @return mixed
     */
    function anna_with_error_handling($callback, $operation_name = 'operation', $default_response = null)
    {
        $benchmark_name = 'anna_' . $operation_name;
        
        try {
            // Start performance monitoring
            if (anna_is_performance_monitoring_enabled()) {
                anna_start_benchmark($benchmark_name);
                anna_memory_snapshot('before_' . $operation_name);
            }
            
            // Execute the callback
            $result = $callback();
            
            // End performance monitoring
            if (anna_is_performance_monitoring_enabled()) {
                anna_memory_snapshot('after_' . $operation_name);
                anna_end_benchmark($benchmark_name);
            }
            
            return $result;
            
        } catch (Exception $e) {
            // Log the error
            $error_id = anna_log_error(
                'Operation failed: ' . $e->getMessage(),
                'ERROR',
                'SYSTEM',
                [
                    'operation' => $operation_name,
                    'exception' => get_class($e),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString()
                ]
            );
            
            // Return default response or formatted error
            if ($default_response !== null) {
                return $default_response;
            }
            
            return anna_format_error_response($e->getMessage(), $error_id);
        }
    }
}