<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Error Handling Setup Script
 * 
 * This script sets up the error handling and debugging infrastructure
 * for the Anna module including database tables and default settings.
 */

/**
 * Install error handling tables and settings
 */
function install_anna_error_handling()
{
    $CI = &get_instance();
    
    // Create error log table
    $error_log_table = db_prefix() . 'anna_error_log';
    if (!$CI->db->table_exists($error_log_table)) {
        $sql = "CREATE TABLE `{$error_log_table}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `error_id` varchar(32) NOT NULL,
            `timestamp` datetime NOT NULL,
            `level` varchar(20) NOT NULL,
            `category` varchar(20) NOT NULL,
            `message` text NOT NULL,
            `staff_id` int(11) DEFAULT NULL,
            `ip_address` varchar(45) DEFAULT NULL,
            `user_agent` text DEFAULT NULL,
            `request_uri` text DEFAULT NULL,
            `context` longtext DEFAULT NULL,
            `stack_trace` longtext DEFAULT NULL,
            `memory_usage` bigint(20) DEFAULT NULL,
            `peak_memory` bigint(20) DEFAULT NULL,
            `execution_time` decimal(10,6) DEFAULT NULL,
            PRIMARY KEY (`id`),
            KEY `error_id` (`error_id`),
            KEY `timestamp` (`timestamp`),
            KEY `level` (`level`),
            KEY `category` (`category`),
            KEY `staff_id` (`staff_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
        
        $CI->db->query($sql);
    }
    
    // Create debug sessions table
    $debug_sessions_table = db_prefix() . 'anna_debug_sessions';
    if (!$CI->db->table_exists($debug_sessions_table)) {
        $sql = "CREATE TABLE `{$debug_sessions_table}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `session_name` varchar(100) NOT NULL,
            `staff_id` int(11) DEFAULT NULL,
            `execution_time` decimal(10,6) DEFAULT NULL,
            `memory_used` bigint(20) DEFAULT NULL,
            `events_count` int(11) DEFAULT NULL,
            `session_data` longtext DEFAULT NULL,
            `created_at` datetime NOT NULL,
            PRIMARY KEY (`id`),
            KEY `session_name` (`session_name`),
            KEY `staff_id` (`staff_id`),
            KEY `created_at` (`created_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
        
        $CI->db->query($sql);
    }
    
    // Set default options
    $default_options = [
        'anna_debug_enabled' => '0',
        'anna_performance_monitoring' => '0',
        'anna_save_debug_sessions' => '0',
        'anna_error_reporting_email' => '',
        'anna_log_retention_days' => '30'
    ];
    
    foreach ($default_options as $option => $value) {
        if (!get_option($option)) {
            add_option($option, $value);
        }
    }
    
    return true;
}

/**
 * Uninstall error handling infrastructure
 */
function uninstall_anna_error_handling()
{
    $CI = &get_instance();
    
    // Drop tables
    $tables = [
        db_prefix() . 'anna_error_log',
        db_prefix() . 'anna_debug_sessions'
    ];
    
    foreach ($tables as $table) {
        if ($CI->db->table_exists($table)) {
            $CI->db->query("DROP TABLE `{$table}`");
        }
    }
    
    // Remove options
    $options = [
        'anna_debug_enabled',
        'anna_performance_monitoring',
        'anna_save_debug_sessions',
        'anna_error_reporting_email',
        'anna_log_retention_days'
    ];
    
    foreach ($options as $option) {
        delete_option($option);
    }
    
    return true;
}

/**
 * Check if error handling is properly installed
 */
function is_anna_error_handling_installed()
{
    $CI = &get_instance();
    
    $required_tables = [
        db_prefix() . 'anna_error_log',
        db_prefix() . 'anna_debug_sessions'
    ];
    
    foreach ($required_tables as $table) {
        if (!$CI->db->table_exists($table)) {
            return false;
        }
    }
    
    return true;
}

/**
 * Get error handling status
 */
function get_anna_error_handling_status()
{
    $CI = &get_instance();
    
    $status = [
        'installed' => is_anna_error_handling_installed(),
        'debug_enabled' => get_option('anna_debug_enabled', '0') === '1',
        'performance_monitoring' => get_option('anna_performance_monitoring', '0') === '1',
        'error_reporting_email' => get_option('anna_error_reporting_email', ''),
        'log_retention_days' => get_option('anna_log_retention_days', '30'),
        'tables' => []
    ];
    
    $tables = [
        'error_log' => db_prefix() . 'anna_error_log',
        'debug_sessions' => db_prefix() . 'anna_debug_sessions'
    ];
    
    foreach ($tables as $name => $table) {
        $status['tables'][$name] = [
            'exists' => $CI->db->table_exists($table),
            'name' => $table
        ];
        
        if ($status['tables'][$name]['exists']) {
            $count = $CI->db->count_all($table);
            $status['tables'][$name]['record_count'] = $count;
        }
    }
    
    return $status;
}

/**
 * Run error handling diagnostics
 */
function run_anna_error_handling_diagnostics()
{
    $CI = &get_instance();
    
    $diagnostics = [
        'timestamp' => date('Y-m-d H:i:s'),
        'php_version' => PHP_VERSION,
        'memory_limit' => ini_get('memory_limit'),
        'max_execution_time' => ini_get('max_execution_time'),
        'log_errors' => ini_get('log_errors') ? 'On' : 'Off',
        'display_errors' => ini_get('display_errors') ? 'On' : 'Off',
        'error_reporting' => error_reporting(),
        'anna_status' => get_anna_error_handling_status(),
        'recent_errors' => []
    ];
    
    // Get recent errors if table exists
    if ($CI->db->table_exists(db_prefix() . 'anna_error_log')) {
        $CI->db->select('error_id, timestamp, level, category, message');
        $CI->db->from(db_prefix() . 'anna_error_log');
        $CI->db->order_by('timestamp', 'DESC');
        $CI->db->limit(5);
        
        $diagnostics['recent_errors'] = $CI->db->get()->result_array();
    }
    
    return $diagnostics;
}

/**
 * Test error handling functionality
 */
function test_anna_error_handling()
{
    $CI = &get_instance();
    
    if (!is_anna_error_handling_installed()) {
        return [
            'success' => false,
            'message' => 'Error handling not installed'
        ];
    }
    
    $CI->load->library('anna_error_handler');
    $CI->load->library('anna_debug');
    
    try {
        // Test error logging
        $error_id = $CI->anna_error_handler->logError(
            'Test error message',
            'ERROR',
            'SYSTEM',
            ['test' => true, 'timestamp' => time()]
        );
        
        // Test debug session
        $CI->anna_debug->startSession('test_session');
        $CI->anna_debug->logDebugEvent('test_session', 'Test event', ['data' => 'test']);
        $CI->anna_debug->endSession('test_session');
        
        return [
            'success' => true,
            'message' => 'Error handling test completed successfully',
            'error_id' => $error_id
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Error handling test failed: ' . $e->getMessage()
        ];
    }
}

/**
 * Auto-run installation check
 */
if (!is_anna_error_handling_installed()) {
    install_anna_error_handling();
}