<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Professional Debug and Performance Monitoring for Anna Module
 * 
 * Provides comprehensive debugging, performance monitoring, and profiling
 * capabilities for production environments with minimal overhead.
 */
class Anna_debug
{
    private $CI;
    private $debug_enabled;
    private $performance_enabled;
    private $start_time;
    private $benchmarks = [];
    private $queries = [];
    private $memory_snapshots = [];
    private $debug_data = [];
    private $profiling_data = [];
    
    public function __construct()
    {
        $this->CI = &get_instance();
        $this->debug_enabled = get_option('anna_debug_enabled', '0') === '1';
        $this->performance_enabled = get_option('anna_performance_monitoring', '0') === '1';
        $this->start_time = microtime(true);
        
        if ($this->debug_enabled || $this->performance_enabled) {
            $this->initializeDebugSession();
        }
    }
    
    /**
     * Start a debug session
     */
    public function startSession($session_name = 'anna_debug')
    {
        if (!$this->debug_enabled) return;
        
        $this->debug_data[$session_name] = [
            'start_time' => microtime(true),
            'start_memory' => memory_get_usage(true),
            'events' => [],
            'queries' => [],
            'benchmarks' => []
        ];
        
        $this->logDebugEvent($session_name, 'Session started');
    }
    
    /**
     * End a debug session
     */
    public function endSession($session_name = 'anna_debug')
    {
        if (!$this->debug_enabled || !isset($this->debug_data[$session_name])) return;
        
        $session = &$this->debug_data[$session_name];
        $session['end_time'] = microtime(true);
        $session['end_memory'] = memory_get_usage(true);
        $session['execution_time'] = $session['end_time'] - $session['start_time'];
        $session['memory_used'] = $session['end_memory'] - $session['start_memory'];
        
        $this->logDebugEvent($session_name, 'Session ended', [
            'execution_time' => $session['execution_time'],
            'memory_used' => $session['memory_used']
        ]);
        
        // Save session data if needed
        if (get_option('anna_save_debug_sessions', '0') === '1') {
            $this->saveDebugSession($session_name);
        }
    }
    
    /**
     * Log a debug event
     */
    public function logDebugEvent($session_name, $event_name, $data = [])
    {
        if (!$this->debug_enabled) return;
        
        if (!isset($this->debug_data[$session_name])) {
            $this->startSession($session_name);
        }
        
        $this->debug_data[$session_name]['events'][] = [
            'timestamp' => microtime(true),
            'event' => $event_name,
            'data' => $data,
            'memory' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true)
        ];
        
        // Also log to CodeIgniter's log
        log_message('debug', "[ANNA:DEBUG:{$session_name}] {$event_name}: " . json_encode($data));
    }
    
    /**
     * Start a performance benchmark
     */
    public function startBenchmark($name)
    {
        if (!$this->performance_enabled) return;
        
        $this->benchmarks[$name] = [
            'start_time' => microtime(true),
            'start_memory' => memory_get_usage(true),
            'queries_start' => $this->getQueryCount()
        ];
    }
    
    /**
     * End a performance benchmark
     */
    public function endBenchmark($name)
    {
        if (!$this->performance_enabled || !isset($this->benchmarks[$name])) return;
        
        $benchmark = &$this->benchmarks[$name];
        $benchmark['end_time'] = microtime(true);
        $benchmark['end_memory'] = memory_get_usage(true);
        $benchmark['queries_end'] = $this->getQueryCount();
        
        $benchmark['execution_time'] = $benchmark['end_time'] - $benchmark['start_time'];
        $benchmark['memory_used'] = $benchmark['end_memory'] - $benchmark['start_memory'];
        $benchmark['queries_executed'] = $benchmark['queries_end'] - $benchmark['queries_start'];
        
        // Log performance data
        $this->logPerformanceData($name, $benchmark);
        
        return $benchmark;
    }
    
    /**
     * Monitor database queries
     */
    public function monitorQuery($query, $execution_time = null, $result_count = null)
    {
        if (!$this->performance_enabled) return;
        
        $this->queries[] = [
            'query' => $query,
            'execution_time' => $execution_time,
            'result_count' => $result_count,
            'timestamp' => microtime(true),
            'memory' => memory_get_usage(true)
        ];
    }
    
    /**
     * Take a memory snapshot
     */
    public function takeMemorySnapshot($label = null)
    {
        if (!$this->performance_enabled) return;
        
        $this->memory_snapshots[] = [
            'label' => $label ?: 'Snapshot ' . (count($this->memory_snapshots) + 1),
            'timestamp' => microtime(true),
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true),
            'memory_real' => memory_get_usage(false)
        ];
    }
    
    /**
     * Analyze performance bottlenecks
     */
    public function analyzePerformance()
    {
        if (!$this->performance_enabled) return [];
        
        $analysis = [
            'total_execution_time' => microtime(true) - $this->start_time,
            'peak_memory_usage' => memory_get_peak_usage(true),
            'query_count' => count($this->queries),
            'slow_queries' => [],
            'memory_leaks' => [],
            'benchmarks' => $this->benchmarks,
            'recommendations' => []
        ];
        
        // Find slow queries
        foreach ($this->queries as $query) {
            if ($query['execution_time'] && $query['execution_time'] > 0.1) { // 100ms threshold
                $analysis['slow_queries'][] = $query;
            }
        }
        
        // Check for memory leaks
        $this->detectMemoryLeaks($analysis);
        
        // Generate recommendations
        $this->generatePerformanceRecommendations($analysis);
        
        return $analysis;
    }
    
    /**
     * Get debug information for current request
     */
    public function getDebugInfo()
    {
        if (!$this->debug_enabled) return [];
        
        return [
            'debug_enabled' => $this->debug_enabled,
            'performance_enabled' => $this->performance_enabled,
            'execution_time' => microtime(true) - $this->start_time,
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true),
            'sessions' => $this->debug_data,
            'benchmarks' => $this->benchmarks,
            'queries' => $this->queries,
            'memory_snapshots' => $this->memory_snapshots,
            'system_info' => $this->getSystemInfo()
        ];
    }
    
    /**
     * Generate a debug report
     */
    public function generateDebugReport($format = 'json')
    {
        $report = [
            'timestamp' => date('Y-m-d H:i:s'),
            'request_uri' => $_SERVER['REQUEST_URI'] ?? 'CLI',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'CLI',
            'staff_id' => get_staff_user_id(),
            'debug_info' => $this->getDebugInfo(),
            'performance_analysis' => $this->analyzePerformance(),
            'environment' => ENVIRONMENT,
            'php_version' => PHP_VERSION,
            'codeigniter_version' => CI_VERSION
        ];
        
        switch ($format) {
            case 'html':
                return $this->formatDebugReportHTML($report);
            case 'text':
                return $this->formatDebugReportText($report);
            default:
                return json_encode($report, JSON_PRETTY_PRINT);
        }
    }
    
    /**
     * Save debug session to database
     */
    private function saveDebugSession($session_name)
    {
        $this->ensureDebugTable();
        
        $session = $this->debug_data[$session_name];
        
        $this->CI->db->insert(db_prefix() . 'anna_debug_sessions', [
            'session_name' => $session_name,
            'staff_id' => get_staff_user_id(),
            'execution_time' => $session['execution_time'],
            'memory_used' => $session['memory_used'],
            'events_count' => count($session['events']),
            'session_data' => json_encode($session),
            'created_at' => date('Y-m-d H:i:s')
        ]);
    }
    
    /**
     * Initialize debug session
     */
    private function initializeDebugSession()
    {
        // Hook into CodeIgniter's benchmark
        if (class_exists('CI_Benchmark')) {
            $this->CI->benchmark->mark('anna_start');
        }
        
        // Start default session
        $this->startSession();
        
        // Take initial memory snapshot
        $this->takeMemorySnapshot('Initial');
    }
    
    /**
     * Get current query count
     */
    private function getQueryCount()
    {
        if (isset($this->CI->db->query_times)) {
            return count($this->CI->db->query_times);
        }
        return 0;
    }
    
    /**
     * Log performance data
     */
    private function logPerformanceData($name, $benchmark)
    {
        if ($benchmark['execution_time'] > 0.1) { // Only log slow operations
            log_message('debug', "[ANNA:PERFORMANCE] {$name}: " . 
                       "Time: {$benchmark['execution_time']}s, " .
                       "Memory: " . $this->formatBytes($benchmark['memory_used']) . ", " .
                       "Queries: {$benchmark['queries_executed']}");
        }
    }
    
    /**
     * Detect memory leaks
     */
    private function detectMemoryLeaks(&$analysis)
    {
        if (count($this->memory_snapshots) < 2) return;
        
        $significant_increase = 1024 * 1024; // 1MB threshold
        
        for ($i = 1; $i < count($this->memory_snapshots); $i++) {
            $current = $this->memory_snapshots[$i];
            $previous = $this->memory_snapshots[$i - 1];
            
            $increase = $current['memory_usage'] - $previous['memory_usage'];
            
            if ($increase > $significant_increase) {
                $analysis['memory_leaks'][] = [
                    'between' => $previous['label'] . ' -> ' . $current['label'],
                    'increase' => $increase,
                    'formatted_increase' => $this->formatBytes($increase)
                ];
            }
        }
    }
    
    /**
     * Generate performance recommendations
     */
    private function generatePerformanceRecommendations(&$analysis)
    {
        $recommendations = [];
        
        // Check execution time
        if ($analysis['total_execution_time'] > 2) {
            $recommendations[] = 'Consider optimizing slow operations or implementing caching';
        }
        
        // Check memory usage
        if ($analysis['peak_memory_usage'] > 64 * 1024 * 1024) { // 64MB
            $recommendations[] = 'High memory usage detected. Consider optimizing data structures';
        }
        
        // Check query count
        if ($analysis['query_count'] > 20) {
            $recommendations[] = 'High number of database queries. Consider query optimization or caching';
        }
        
        // Check slow queries
        if (count($analysis['slow_queries']) > 0) {
            $recommendations[] = 'Slow queries detected. Consider adding indexes or optimizing queries';
        }
        
        // Check memory leaks
        if (count($analysis['memory_leaks']) > 0) {
            $recommendations[] = 'Potential memory leaks detected. Review resource cleanup';
        }
        
        $analysis['recommendations'] = $recommendations;
    }
    
    /**
     * Get system information
     */
    private function getSystemInfo()
    {
        return [
            'php_version' => PHP_VERSION,
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'post_max_size' => ini_get('post_max_size'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            'loaded_extensions' => get_loaded_extensions()
        ];
    }
    
    /**
     * Format debug report as HTML
     */
    private function formatDebugReportHTML($report)
    {
        $html = '<html><head><title>Anna Debug Report</title>';
        $html .= '<style>body{font-family:Arial,sans-serif;margin:20px;}';
        $html .= '.section{margin-bottom:20px;padding:15px;border:1px solid #ccc;border-radius:5px;}';
        $html .= '.warning{background-color:#fff3cd;color:#856404;}';
        $html .= '.error{background-color:#f8d7da;color:#721c24;}';
        $html .= 'pre{background-color:#f8f9fa;padding:10px;border-radius:3px;overflow-x:auto;}';
        $html .= '</style></head><body>';
        
        $html .= '<h1>Anna Debug Report</h1>';
        $html .= '<div class="section"><h2>Summary</h2>';
        $html .= '<p>Generated: ' . $report['timestamp'] . '</p>';
        $html .= '<p>Request: ' . htmlspecialchars($report['request_uri']) . '</p>';
        $html .= '<p>Staff ID: ' . $report['staff_id'] . '</p>';
        $html .= '</div>';
        
        if (!empty($report['performance_analysis']['recommendations'])) {
            $html .= '<div class="section warning"><h2>Recommendations</h2><ul>';
            foreach ($report['performance_analysis']['recommendations'] as $rec) {
                $html .= '<li>' . htmlspecialchars($rec) . '</li>';
            }
            $html .= '</ul></div>';
        }
        
        $html .= '<div class="section"><h2>Performance Analysis</h2>';
        $html .= '<pre>' . htmlspecialchars(json_encode($report['performance_analysis'], JSON_PRETTY_PRINT)) . '</pre>';
        $html .= '</div>';
        
        $html .= '<div class="section"><h2>Full Debug Data</h2>';
        $html .= '<pre>' . htmlspecialchars(json_encode($report['debug_info'], JSON_PRETTY_PRINT)) . '</pre>';
        $html .= '</div>';
        
        $html .= '</body></html>';
        
        return $html;
    }
    
    /**
     * Format debug report as text
     */
    private function formatDebugReportText($report)
    {
        $text = "ANNA DEBUG REPORT\n";
        $text .= "=================\n\n";
        $text .= "Generated: " . $report['timestamp'] . "\n";
        $text .= "Request: " . $report['request_uri'] . "\n";
        $text .= "Staff ID: " . $report['staff_id'] . "\n\n";
        
        if (!empty($report['performance_analysis']['recommendations'])) {
            $text .= "RECOMMENDATIONS:\n";
            foreach ($report['performance_analysis']['recommendations'] as $rec) {
                $text .= "- " . $rec . "\n";
            }
            $text .= "\n";
        }
        
        $text .= "PERFORMANCE ANALYSIS:\n";
        $text .= json_encode($report['performance_analysis'], JSON_PRETTY_PRINT) . "\n\n";
        
        $text .= "DEBUG DATA:\n";
        $text .= json_encode($report['debug_info'], JSON_PRETTY_PRINT) . "\n";
        
        return $text;
    }
    
    /**
     * Format bytes
     */
    private function formatBytes($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        return round($bytes, 2) . ' ' . $units[$i];
    }
    
    /**
     * Ensure debug table exists
     */
    private function ensureDebugTable()
    {
        if (!$this->CI->db->table_exists(db_prefix() . 'anna_debug_sessions')) {
            $sql = "CREATE TABLE `" . db_prefix() . "anna_debug_sessions` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `session_name` varchar(100) NOT NULL,
                `staff_id` int(11) DEFAULT NULL,
                `execution_time` decimal(10,6) DEFAULT NULL,
                `memory_used` bigint(20) DEFAULT NULL,
                `events_count` int(11) DEFAULT NULL,
                `session_data` longtext DEFAULT NULL,
                `created_at` datetime NOT NULL,
                PRIMARY KEY (`id`),
                KEY `session_name` (`session_name`),
                KEY `staff_id` (`staff_id`),
                KEY `created_at` (`created_at`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
            
            $this->CI->db->query($sql);
        }
    }
    
    /**
     * Cleanup old debug sessions
     */
    public function cleanupOldSessions($days = 7)
    {
        $this->ensureDebugTable();
        
        $cutoff = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $this->CI->db->where('created_at <', $cutoff);
        $this->CI->db->delete(db_prefix() . 'anna_debug_sessions');
        
        return $this->CI->db->affected_rows();
    }
}