<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Professional Error Handler for Anna Module
 * 
 * Provides comprehensive error handling, logging, and debugging capabilities
 * for production environments with proper error categorization and alerting.
 */
class Anna_error_handler
{
    private $CI;
    private $log_file_path;
    private $debug_mode;
    private $error_reporting_email;
    private $error_counts = [];
    private $rate_limit_window = 300; // 5 minutes
    private $max_errors_per_window = 50;
    
    // Error levels
    const LEVEL_DEBUG = 'DEBUG';
    const LEVEL_INFO = 'INFO';
    const LEVEL_WARNING = 'WARNING';
    const LEVEL_ERROR = 'ERROR';
    const LEVEL_CRITICAL = 'CRITICAL';
    
    // Error categories
    const CATEGORY_API = 'API';
    const CATEGORY_DATABASE = 'DATABASE';
    const CATEGORY_AUTHENTICATION = 'AUTH';
    const CATEGORY_VALIDATION = 'VALIDATION';
    const CATEGORY_SYSTEM = 'SYSTEM';
    const CATEGORY_EXTERNAL = 'EXTERNAL';
    
    public function __construct()
    {
        $this->CI = &get_instance();
        $this->log_file_path = APPPATH . 'logs/anna_errors.log';
        $this->debug_mode = defined('ENVIRONMENT') && ENVIRONMENT === 'development';
        $this->error_reporting_email = get_option('anna_error_reporting_email');
        
        // Ensure log directory exists
        $this->ensureLogDirectory();
        
        // Set up error handlers
        $this->setupErrorHandlers();
    }
    
    /**
     * Log error with proper categorization and context
     */
    public function logError($message, $level = self::LEVEL_ERROR, $category = self::CATEGORY_SYSTEM, $context = [])
    {
        $error_id = $this->generateErrorId();
        $timestamp = date('Y-m-d H:i:s');
        $staff_id = get_staff_user_id() ?: 'system';
        $ip_address = $this->CI->input->ip_address();
        $user_agent = $this->CI->input->user_agent();
        $request_uri = $_SERVER['REQUEST_URI'] ?? 'CLI';
        
        // Check rate limiting
        if ($this->isRateLimited($level, $category)) {
            return $error_id;
        }
        
        // Prepare error data
        $error_data = [
            'error_id' => $error_id,
            'timestamp' => $timestamp,
            'level' => $level,
            'category' => $category,
            'message' => $message,
            'staff_id' => $staff_id,
            'ip_address' => $ip_address,
            'user_agent' => $user_agent,
            'request_uri' => $request_uri,
            'context' => $context,
            'stack_trace' => $this->getStackTrace(),
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true),
            'execution_time' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']
        ];
        
        // Log to file
        $this->writeToLogFile($error_data);
        
        // Log to database
        $this->logToDatabase($error_data);
        
        // Send alerts for critical errors
        if ($level === self::LEVEL_CRITICAL) {
            $this->sendCriticalErrorAlert($error_data);
        }
        
        // Log to CodeIgniter's log system
        $ci_level = $this->convertToCILogLevel($level);
        log_message($ci_level, "[ANNA:{$category}] {$message}");
        
        return $error_id;
    }
    
    /**
     * Handle API errors with proper response formatting
     */
    public function handleApiError($error, $http_code = 500, $additional_context = [])
    {
        $error_id = $this->generateErrorId();
        $level = $http_code >= 500 ? self::LEVEL_ERROR : self::LEVEL_WARNING;
        
        $context = array_merge([
            'http_code' => $http_code,
            'endpoint' => $_SERVER['REQUEST_URI'] ?? 'unknown',
            'method' => $_SERVER['REQUEST_METHOD'] ?? 'unknown',
            'post_data' => $this->sanitizePostData(),
            'headers' => $this->getRequestHeaders()
        ], $additional_context);
        
        $this->logError($error, $level, self::CATEGORY_API, $context);
        
        // Return standardized error response
        return [
            'success' => false,
            'error' => $this->debug_mode ? $error : 'An error occurred while processing your request.',
            'error_id' => $error_id,
            'timestamp' => date('c'),
            'debug_info' => $this->debug_mode ? $context : null
        ];
    }
    
    /**
     * Handle database errors
     */
    public function handleDatabaseError($error, $query = null, $additional_context = [])
    {
        $error_id = $this->generateErrorId();
        
        $context = array_merge([
            'query' => $this->sanitizeQuery($query),
            'db_error' => $this->CI->db->error(),
            'last_query' => $this->CI->db->last_query()
        ], $additional_context);
        
        $this->logError($error, self::LEVEL_ERROR, self::CATEGORY_DATABASE, $context);
        
        return $error_id;
    }
    
    /**
     * Handle external API errors (curl, API calls)
     */
    public function handleExternalError($error, $url = null, $response_code = null, $additional_context = [])
    {
        $error_id = $this->generateErrorId();
        
        $context = array_merge([
            'url' => $url,
            'response_code' => $response_code,
            'curl_error' => curl_error($this->CI->curl ?? null),
            'timeout' => ini_get('default_socket_timeout')
        ], $additional_context);
        
        $this->logError($error, self::LEVEL_ERROR, self::CATEGORY_EXTERNAL, $context);
        
        return $error_id;
    }
    
    /**
     * Handle validation errors
     */
    public function handleValidationError($errors, $additional_context = [])
    {
        $error_id = $this->generateErrorId();
        
        $context = array_merge([
            'validation_errors' => $errors,
            'form_data' => $this->sanitizePostData()
        ], $additional_context);
        
        $this->logError('Validation failed', self::LEVEL_WARNING, self::CATEGORY_VALIDATION, $context);
        
        return [
            'success' => false,
            'error' => 'Validation failed',
            'validation_errors' => $errors,
            'error_id' => $error_id
        ];
    }
    
    /**
     * Get error statistics for monitoring
     */
    public function getErrorStats($hours = 24)
    {
        $since = date('Y-m-d H:i:s', strtotime("-{$hours} hours"));
        
        $this->CI->db->select('level, category, COUNT(*) as count');
        $this->CI->db->from(db_prefix() . 'anna_error_log');
        $this->CI->db->where('timestamp >=', $since);
        $this->CI->db->group_by(['level', 'category']);
        
        $stats = $this->CI->db->get()->result_array();
        
        return [
            'period' => $hours . ' hours',
            'total_errors' => array_sum(array_column($stats, 'count')),
            'breakdown' => $stats,
            'last_updated' => date('Y-m-d H:i:s')
        ];
    }
    
    /**
     * Clean up old error logs
     */
    public function cleanupOldLogs($days = 30)
    {
        $cutoff = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $this->CI->db->where('timestamp <', $cutoff);
        $this->CI->db->delete(db_prefix() . 'anna_error_log');
        
        $deleted = $this->CI->db->affected_rows();
        
        // Also cleanup log files
        $this->cleanupLogFiles($days);
        
        return $deleted;
    }
    
    /**
     * Private helper methods
     */
    private function setupErrorHandlers()
    {
        // Set up PHP error handler for Anna-specific errors
        set_error_handler([$this, 'phpErrorHandler'], E_ALL);
        
        // Set up exception handler
        set_exception_handler([$this, 'exceptionHandler']);
    }
    
    private function ensureLogDirectory()
    {
        $log_dir = dirname($this->log_file_path);
        if (!is_dir($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
        
        // Ensure error log table exists
        $this->ensureErrorLogTable();
    }
    
    private function ensureErrorLogTable()
    {
        if (!$this->CI->db->table_exists(db_prefix() . 'anna_error_log')) {
            $sql = "CREATE TABLE `" . db_prefix() . "anna_error_log` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `error_id` varchar(32) NOT NULL,
                `timestamp` datetime NOT NULL,
                `level` varchar(20) NOT NULL,
                `category` varchar(20) NOT NULL,
                `message` text NOT NULL,
                `staff_id` int(11) DEFAULT NULL,
                `ip_address` varchar(45) DEFAULT NULL,
                `user_agent` text DEFAULT NULL,
                `request_uri` text DEFAULT NULL,
                `context` longtext DEFAULT NULL,
                `stack_trace` longtext DEFAULT NULL,
                `memory_usage` bigint(20) DEFAULT NULL,
                `peak_memory` bigint(20) DEFAULT NULL,
                `execution_time` decimal(10,6) DEFAULT NULL,
                PRIMARY KEY (`id`),
                KEY `error_id` (`error_id`),
                KEY `timestamp` (`timestamp`),
                KEY `level` (`level`),
                KEY `category` (`category`),
                KEY `staff_id` (`staff_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
            
            $this->CI->db->query($sql);
        }
    }
    
    private function generateErrorId()
    {
        return substr(md5(uniqid(mt_rand(), true)), 0, 16);
    }
    
    private function isRateLimited($level, $category)
    {
        $key = $level . '_' . $category;
        $current_time = time();
        $window_start = $current_time - $this->rate_limit_window;
        
        // Clean old entries
        $this->error_counts[$key] = array_filter(
            $this->error_counts[$key] ?? [],
            function($timestamp) use ($window_start) {
                return $timestamp > $window_start;
            }
        );
        
        // Check if we're over the limit
        if (count($this->error_counts[$key]) >= $this->max_errors_per_window) {
            return true;
        }
        
        // Add current error
        $this->error_counts[$key][] = $current_time;
        
        return false;
    }
    
    private function getStackTrace()
    {
        $trace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS);
        $formatted_trace = [];
        
        foreach ($trace as $i => $frame) {
            if (isset($frame['file'])) {
                $formatted_trace[] = sprintf(
                    "#%d %s(%d): %s%s%s()",
                    $i,
                    $frame['file'],
                    $frame['line'],
                    $frame['class'] ?? '',
                    $frame['type'] ?? '',
                    $frame['function']
                );
            }
        }
        
        return implode("\n", $formatted_trace);
    }
    
    private function writeToLogFile($error_data)
    {
        $log_entry = sprintf(
            "[%s] [%s:%s] %s | ID: %s | Staff: %s | IP: %s | Memory: %s | Context: %s\n",
            $error_data['timestamp'],
            $error_data['level'],
            $error_data['category'],
            $error_data['message'],
            $error_data['error_id'],
            $error_data['staff_id'],
            $error_data['ip_address'],
            $this->formatBytes($error_data['memory_usage']),
            json_encode($error_data['context'])
        );
        
        file_put_contents($this->log_file_path, $log_entry, FILE_APPEND | LOCK_EX);
    }
    
    private function logToDatabase($error_data)
    {
        try {
            $this->CI->db->insert(db_prefix() . 'anna_error_log', [
                'error_id' => $error_data['error_id'],
                'timestamp' => $error_data['timestamp'],
                'level' => $error_data['level'],
                'category' => $error_data['category'],
                'message' => $error_data['message'],
                'staff_id' => $error_data['staff_id'],
                'ip_address' => $error_data['ip_address'],
                'user_agent' => $error_data['user_agent'],
                'request_uri' => $error_data['request_uri'],
                'context' => json_encode($error_data['context']),
                'stack_trace' => $error_data['stack_trace'],
                'memory_usage' => $error_data['memory_usage'],
                'peak_memory' => $error_data['peak_memory'],
                'execution_time' => $error_data['execution_time']
            ]);
        } catch (Exception $e) {
            // Fallback to file logging if database fails
            error_log("Failed to log to database: " . $e->getMessage());
        }
    }
    
    private function sendCriticalErrorAlert($error_data)
    {
        if (!$this->error_reporting_email) {
            return;
        }
        
        $subject = "CRITICAL ERROR - Anna Module - " . $error_data['error_id'];
        $body = "A critical error has occurred in the Anna module:\n\n";
        $body .= "Error ID: " . $error_data['error_id'] . "\n";
        $body .= "Timestamp: " . $error_data['timestamp'] . "\n";
        $body .= "Message: " . $error_data['message'] . "\n";
        $body .= "Category: " . $error_data['category'] . "\n";
        $body .= "Staff ID: " . $error_data['staff_id'] . "\n";
        $body .= "IP Address: " . $error_data['ip_address'] . "\n";
        $body .= "Request URI: " . $error_data['request_uri'] . "\n";
        $body .= "Memory Usage: " . $this->formatBytes($error_data['memory_usage']) . "\n\n";
        $body .= "Stack Trace:\n" . $error_data['stack_trace'] . "\n\n";
        $body .= "Context:\n" . json_encode($error_data['context'], JSON_PRETTY_PRINT);
        
        // Use CodeIgniter's email library
        $this->CI->load->library('email');
        $this->CI->email->from(get_option('smtp_email'), 'Anna Error Reporter');
        $this->CI->email->to($this->error_reporting_email);
        $this->CI->email->subject($subject);
        $this->CI->email->message($body);
        $this->CI->email->send();
    }
    
    private function convertToCILogLevel($level)
    {
        switch ($level) {
            case self::LEVEL_DEBUG:
                return 'debug';
            case self::LEVEL_INFO:
                return 'info';
            case self::LEVEL_WARNING:
                return 'debug';
            case self::LEVEL_ERROR:
                return 'error';
            case self::LEVEL_CRITICAL:
                return 'error';
            default:
                return 'debug';
        }
    }
    
    private function sanitizePostData()
    {
        $post = $_POST;
        
        // Remove sensitive data
        $sensitive_keys = ['password', 'token', 'auth_key', 'api_key', 'secret'];
        foreach ($sensitive_keys as $key) {
            if (isset($post[$key])) {
                $post[$key] = '[REDACTED]';
            }
        }
        
        return $post;
    }
    
    private function sanitizeQuery($query)
    {
        if (!$query) return null;
        
        // Remove sensitive data from queries
        $patterns = [
            '/password\s*=\s*[\'"][^\'"`]*[\'"]/',
            '/token\s*=\s*[\'"][^\'"`]*[\'"]/',
            '/api_key\s*=\s*[\'"][^\'"`]*[\'"]/'
        ];
        
        return preg_replace($patterns, '[REDACTED]', $query);
    }
    
    private function getRequestHeaders()
    {
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $header = str_replace('_', '-', substr($key, 5));
                $headers[$header] = $value;
            }
        }
        
        // Remove authorization headers
        unset($headers['AUTHORIZATION']);
        
        return $headers;
    }
    
    private function formatBytes($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        return round($bytes, 2) . ' ' . $units[$i];
    }
    
    private function cleanupLogFiles($days)
    {
        $log_dir = dirname($this->log_file_path);
        $cutoff = time() - ($days * 24 * 60 * 60);
        
        $files = glob($log_dir . '/anna_*.log');
        foreach ($files as $file) {
            if (filemtime($file) < $cutoff) {
                unlink($file);
            }
        }
    }
    
    public function phpErrorHandler($severity, $message, $file, $line)
    {
        if (!(error_reporting() & $severity)) {
            return;
        }
        
        $level = $this->convertPHPErrorLevel($severity);
        $error_message = "PHP Error: {$message} in {$file} on line {$line}";
        
        $this->logError($error_message, $level, self::CATEGORY_SYSTEM, [
            'file' => $file,
            'line' => $line,
            'severity' => $severity
        ]);
    }
    
    public function exceptionHandler($exception)
    {
        $this->logError(
            "Uncaught Exception: " . $exception->getMessage(),
            self::LEVEL_CRITICAL,
            self::CATEGORY_SYSTEM,
            [
                'exception_class' => get_class($exception),
                'file' => $exception->getFile(),
                'line' => $exception->getLine(),
                'trace' => $exception->getTraceAsString()
            ]
        );
    }
    
    private function convertPHPErrorLevel($severity)
    {
        switch ($severity) {
            case E_ERROR:
            case E_USER_ERROR:
                return self::LEVEL_ERROR;
            case E_WARNING:
            case E_USER_WARNING:
                return self::LEVEL_WARNING;
            case E_NOTICE:
            case E_USER_NOTICE:
                return self::LEVEL_INFO;
            default:
                return self::LEVEL_DEBUG;
        }
    }
}