<?php

// Update to modules/anna/models/Anna_model.php
defined('BASEPATH') or exit('No direct script access allowed');


class Anna_model extends App_Model
{
    private $conversation_id = null;

    public function __construct()
    {
        parent::__construct();
        
        // Load error handling helper
        $this->load->helper('anna_error');
    }

    public function process_chat($message, $conversation_id = null)
    {
        // Initialize or get conversation session
        $this->conversation_id = $this->initializeConversation($conversation_id);
        
        // Simple response - this will be replaced with actual API call to Python service
        $response = "I received your message: " . $message;
        
        // Log the interaction
        $this->log_chat_interaction($message, $response, $this->conversation_id);
        
        return $response;
    }

    public function log_chat_interaction($message, $response, $conversation_id)
    {
        // Log to chat log table
        $this->db->insert(db_prefix() . 'anna_chat_log', [
            'staff_id' => get_staff_user_id(),
            'message' => $message,
            'response' => $response,
            'created_at' => date('Y-m-d H:i:s')
        ]);

        // Check if conversation tables exist, create if not
        $this->ensureConversationTablesExist();

        try {
            // Log to conversation messages table
            $this->db->insert(db_prefix() . 'anna_conversations_messages', [
                'conversation_id' => $conversation_id,
                'role' => 'user',
                'content' => $message,
                'created_at' => date('Y-m-d H:i:s')
            ]);

            $this->db->insert(db_prefix() . 'anna_conversations_messages', [
                'conversation_id' => $conversation_id,
                'role' => 'assistant',
                'content' => $response,
                'created_at' => date('Y-m-d H:i:s')
            ]);

            // Update conversation last activity
            $this->db->where('id', $conversation_id);
            $this->db->update(db_prefix() . 'anna_conversations', [
                'last_activity' => date('Y-m-d H:i:s'),
                'message_count' => $this->db->query("SELECT COUNT(*) as count FROM " . db_prefix() . "anna_conversations_messages WHERE conversation_id = ?", [$conversation_id])->row()->count
            ]);
        } catch (Exception $e) {
            // Continue without failing the chat
            anna_handle_db_error(
                'Failed to log conversation: ' . $e->getMessage(),
                $this->db->last_query(),
                ['conversation_id' => $conversation_id]
            );
        }
    }

    public function get_chat_history($staff_id = null, $limit = 10)
    {
        $staff_id = $staff_id ?: get_staff_user_id();
        
        $this->db->select('*');
        $this->db->from(db_prefix() . 'anna_chat_log');
        $this->db->where('staff_id', $staff_id);
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result_array();
    }

    /**
     * Initialize or get existing conversation
     */
    private function initializeConversation($conversation_id = null)
    {
        $staff_id = get_staff_user_id();
        
        // Ensure conversation tables exist
        $this->ensureConversationTablesExist();
        
        try {
            if ($conversation_id) {
                // Verify conversation exists and belongs to user
                $this->db->where('id', $conversation_id);
                $this->db->where('staff_id', $staff_id);
                $conversation = $this->db->get(db_prefix() . 'anna_conversations')->row();
                
                if ($conversation) {
                    return $conversation_id;
                }
            }
            
            // Create new conversation or get the most recent active one
            $this->db->where('staff_id', $staff_id);
            $this->db->where('status', 'active');
            $this->db->order_by('last_activity', 'DESC');
            $this->db->limit(1);
            $recent_conversation = $this->db->get(db_prefix() . 'anna_conversations')->row();
            
            // If recent conversation is less than 2 hours old, continue it
            if ($recent_conversation && strtotime($recent_conversation->last_activity) > (time() - 7200)) {
                return $recent_conversation->id;
            }
            
            // Create new conversation
            $conversation_data = [
                'staff_id' => $staff_id,
                'title' => $this->generateConversationTitle(),
                'status' => 'active',
                'created_at' => date('Y-m-d H:i:s'),
                'last_activity' => date('Y-m-d H:i:s'),
                'message_count' => 0
            ];
            
            $this->db->insert(db_prefix() . 'anna_conversations', $conversation_data);
            $new_conversation_id = $this->db->insert_id();
            
            return $new_conversation_id;
        } catch (Exception $e) {
            // Return null to fall back to non-conversation mode
            anna_handle_db_error(
                'Failed to initialize conversation: ' . $e->getMessage(),
                $this->db->last_query(),
                ['staff_id' => $staff_id, 'conversation_id' => $conversation_id]
            );
            return null;
        }
    }

    /**
     * Generate conversation title based on first message
     */
    private function generateConversationTitle($first_message = null)
    {
        if (!$first_message) {
            return "Chat Session - " . date('M j, H:i');
        }
        
        // Extract topic from first message
        $words = explode(' ', trim($first_message));
        $title_words = array_slice($words, 0, 5);
        $title = implode(' ', $title_words);
        
        if (strlen($title) > 50) {
            $title = substr($title, 0, 47) . '...';
        }
        
        return $title ?: "Chat Session - " . date('M j, H:i');
    }

    /**
     * Get conversation list for a user
     */
    public function getConversationList($staff_id = null, $limit = 10)
    {
        $staff_id = $staff_id ?: get_staff_user_id();
        
        $this->db->select('id, title, created_at, last_activity, message_count, status');
        $this->db->from(db_prefix() . 'anna_conversations');
        $this->db->where('staff_id', $staff_id);
        $this->db->where('status', 'active'); // Only show active conversations
        $this->db->order_by('last_activity', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result_array();
    }

    /**
     * Get full conversation with messages
     */
    public function getFullConversation($conversation_id, $staff_id = null)
    {
        $staff_id = $staff_id ?: get_staff_user_id();
        
        // Get conversation details
        $this->db->where('id', $conversation_id);
        $this->db->where('staff_id', $staff_id);
        $this->db->where('status', 'active'); // Only get active conversations
        $conversation = $this->db->get(db_prefix() . 'anna_conversations')->row_array();
        
        if (!$conversation) {
            return null;
        }
        
        // Get messages
        $this->db->select('role, content, created_at, metadata');
        $this->db->from(db_prefix() . 'anna_conversations_messages');
        $this->db->where('conversation_id', $conversation_id);
        $this->db->order_by('created_at', 'ASC');
        
        $messages = $this->db->get()->result_array();
        
        return [
            'conversation' => $conversation,
            'messages' => $messages
        ];
    }

    /**
     * Ensure conversation tables exist
     */
    private function ensureConversationTablesExist()
    {
        try {
            // Check if conversations table exists
            if (!$this->db->table_exists(db_prefix() . 'anna_conversations')) {
                $this->db->query('CREATE TABLE `' . db_prefix() . "anna_conversations` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `staff_id` int(11) NOT NULL,
                    `title` varchar(255) NOT NULL,
                    `status` enum('active','archived','deleted') DEFAULT 'active',
                    `created_at` datetime NOT NULL,
                    `last_activity` datetime NOT NULL,
                    `message_count` int(11) DEFAULT 0,
                    `summary` text,
                    PRIMARY KEY (`id`),
                    KEY `staff_id` (`staff_id`),
                    KEY `status` (`status`),
                    KEY `last_activity` (`last_activity`)
                ) ENGINE=InnoDB DEFAULT CHARSET=" . $this->db->char_set . ';');
                
            }

            // Check if conversation messages table exists
            if (!$this->db->table_exists(db_prefix() . 'anna_conversations_messages')) {
                $this->db->query('CREATE TABLE `' . db_prefix() . "anna_conversations_messages` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `conversation_id` int(11) NOT NULL,
                    `role` enum('user','assistant','system') NOT NULL,
                    `content` text NOT NULL,
                    `metadata` text,
                    `created_at` datetime NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `conversation_id` (`conversation_id`),
                    KEY `role` (`role`),
                    KEY `created_at` (`created_at`)
                ) ENGINE=InnoDB DEFAULT CHARSET=" . $this->db->char_set . ';');
                
            }
        } catch (Exception $e) {
            // Only log real errors, not "table already exists" warnings
            if (strpos($e->getMessage(), 'already exists') === false) {
            }
        }
    }

    /**
     * Archive old conversations
     */
    public function archiveOldConversations($days_old = 30)
    {
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days_old} days"));
        
        $this->db->where('last_activity <', $cutoff_date);
        $this->db->where('status', 'active');
        $this->db->update(db_prefix() . 'anna_conversations', ['status' => 'archived']);
        
        return $this->db->affected_rows();
    }

    /**
     * Force create a new conversation
     */
    public function createNewConversation($title = null)
    {
        $this->ensureConversationTablesExist();
        
        $staff_id = get_staff_user_id();
        $conversation_data = [
            'staff_id' => $staff_id,
            'title' => $title ?: $this->generateConversationTitle(),
            'status' => 'active',
            'created_at' => date('Y-m-d H:i:s'),
            'last_activity' => date('Y-m-d H:i:s'),
            'message_count' => 0
        ];
        
        $this->db->insert(db_prefix() . 'anna_conversations', $conversation_data);
        $conversation_id = $this->db->insert_id();
        
        
        return $conversation_id;
    }

    /**
     * Delete a conversation
     */
    public function deleteConversation($conversation_id)
    {
        // Ensure tables exist
        $this->ensureConversationTablesExist();
        
        $current_staff_id = get_staff_user_id();
        
        // Debug: Log the attempt
        log_message('debug', "DeleteConversation: ID=$conversation_id, Staff=$current_staff_id, IsAdmin=" . (is_admin() ? 'yes' : 'no'));
        
        // Check if the conversation exists first with table prefix
        $this->db->where('id', $conversation_id);
        $this->db->where('status !=', 'deleted');
        $conversation = $this->db->get(db_prefix() . 'anna_conversations')->row();
        
        // Debug: Log what we found
        log_message('debug', "DeleteConversation: Query result = " . ($conversation ? "Found (owner: {$conversation->staff_id})" : "Not found"));
        
        if (!$conversation) {
            log_message('debug', "DeleteConversation: Conversation not found or already deleted");
            return false; // Conversation doesn't exist or already deleted
        }
        
        // Admin can delete any conversation, users can only delete their own
        $can_delete = is_admin() || $conversation->staff_id == $current_staff_id;
        
        log_message('debug', "DeleteConversation: Can delete = " . ($can_delete ? 'yes' : 'no'));
        
        if (!$can_delete) {
            log_message('debug', "DeleteConversation: Access denied");
            return false; // Access denied
        }
        
        // Perform the deletion
        $this->db->where('id', $conversation_id);
        $update_result = $this->db->update(db_prefix() . 'anna_conversations', ['status' => 'deleted']);
        
        $affected_rows = $this->db->affected_rows();
        log_message('debug', "DeleteConversation: Update result = " . ($update_result ? 'success' : 'failed') . ", Affected rows = $affected_rows");
        
        return $affected_rows > 0;
    }

    /**
     * Rename a conversation
     */
    public function renameConversation($conversation_id, $new_title)
    {
        $staff_id = get_staff_user_id();
        
        $this->db->where('id', $conversation_id);
        $this->db->where('staff_id', $staff_id);
        $this->db->update(db_prefix() . 'anna_conversations', ['title' => $new_title]);
        
        return $this->db->affected_rows() > 0;
    }

    /**
     * Get conversation summary for large context trimming
     */
    public function getConversationSummary($conversation_id)
    {
        try {
            $this->db->select('summary');
            $this->db->from(db_prefix() . 'anna_conversations');
            $this->db->where('id', $conversation_id);
            $result = $this->db->get()->row();
            
            return $result ? $result->summary : null;
        } catch (Exception $e) {
            log_message('error', 'Failed to get conversation summary: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Update conversation summary
     */
    public function updateConversationSummary($conversation_id, $summary)
    {
        try {
            $this->db->where('id', $conversation_id);
            $this->db->update(db_prefix() . 'anna_conversations', ['summary' => $summary]);
            return true;
        } catch (Exception $e) {
            log_message('error', 'Failed to update conversation summary: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get current conversation ID
     */
    public function getCurrentConversationId()
    {
        return $this->conversation_id;
    }
}
