<?php
// modules/anna/views/chat_widget.php
defined('BASEPATH') or exit('No direct script access allowed'); ?>

<!-- Anna Drawer Widget -->
<div id="perfex-anna-widget" class="pc-anna-widget">
    <!-- Floating Action Button -->
    <div class="pc-fab" onclick="toggleAnnaDrawer()">
        <div class="pc-fab-icon">
            <svg width="28" height="28" viewBox="0 0 64 64" fill="currentColor">
                <path d="M22.625 2c0 15.834-8.557 30-20.625 30c12.068 0 20.625 14.167 20.625 30c0-15.833 8.557-30 20.625-30c-12.068 0-20.625-14.166-20.625-30"></path>
                <path d="M47 32c0 7.918-4.277 15-10.313 15C42.723 47 47 54.084 47 62c0-7.916 4.277-15 10.313-15C51.277 47 47 39.918 47 32z"></path>
                <path d="M51.688 2c0 7.917-4.277 15-10.313 15c6.035 0 10.313 7.084 10.313 15c0-7.916 4.277-15 10.313-15c-6.036 0-10.313-7.083-10.313-15"></path>
            </svg>
        </div>
        <div class="pc-fab-notification" id="pc-notification" style="display: none;">1</div>
    </div>
    
    <!-- Overlay -->
    <div class="pc-drawer-overlay" id="pc-drawer-overlay" onclick="toggleAnnaDrawer()"></div>
    
    <!-- Drawer -->
    <div class="pc-drawer" id="pc-drawer">
        <!-- Drawer Header -->
        <div class="pc-drawer-header">
            <div class="pc-drawer-header-main">
                <div class="pc-drawer-brand">
                    <div class="pc-drawer-avatar">
                        <svg width="24" height="24" viewBox="0 0 64 64" fill="currentColor">
                            <path d="M22.625 2c0 15.834-8.557 30-20.625 30c12.068 0 20.625 14.167 20.625 30c0-15.833 8.557-30 20.625-30c-12.068 0-20.625-14.166-20.625-30"></path>
                            <path d="M47 32c0 7.918-4.277 15-10.313 15C42.723 47 47 54.084 47 62c0-7.916 4.277-15 10.313-15C51.277 47 47 39.918 47 32z"></path>
                            <path d="M51.688 2c0 7.917-4.277 15-10.313 15c6.035 0 10.313 7.084 10.313 15c0-7.916 4.277-15 10.313-15c-6.036 0-10.313-7.083-10.313-15"></path>
                        </svg>
                    </div>
                    <div class="pc-drawer-brand-info">
                        <h2 class="pc-drawer-title">Anna</h2>
                        <div class="pc-drawer-status">
                            <div class="pc-status-indicator"></div>
                            <span id="pc-drawer-status-text">AI Assistant Ready</span>
                        </div>
                    </div>
                </div>
                
                <div class="pc-drawer-actions">
                    <button onclick="createDrawerNewChat()" class="pc-drawer-action-btn" title="New Conversation">
                        <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                        </svg>
                    </button>
                    <button onclick="openAnnaChatPage()" class="pc-drawer-action-btn" title="Open Full Chat">
                        <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14"></path>
                        </svg>
                    </button>
                    <button onclick="toggleAnnaDrawer()" class="pc-drawer-close" title="Close">
                        <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
            </div>
            
            <!-- Session Message as Header Extension -->
            <div id="widget-session-message-container" class="pc-drawer-session-bar" style="display: none;">
                <div class="pc-session-content">
                    <svg width="16" height="16" fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"></path>
                    </svg>
                    <span id="widget-session-message-text"></span>
                </div>
            </div>
        </div>
        
        <!-- Chat Content -->
        <div class="pc-drawer-content">
            <!-- Welcome Message -->
            <div class="pc-drawer-welcome" id="pc-drawer-welcome">
                <div class="pc-welcome-icon">
                    <svg width="48" height="48" viewBox="0 0 64 64" fill="currentColor">
                        <path d="M22.625 2c0 15.834-8.557 30-20.625 30c12.068 0 20.625 14.167 20.625 30c0-15.833 8.557-30 20.625-30c-12.068 0-20.625-14.166-20.625-30"></path>
                        <path d="M47 32c0 7.918-4.277 15-10.313 15C42.723 47 47 54.084 47 62c0-7.916 4.277-15 10.313-15C51.277 47 47 39.918 47 32z"></path>
                        <path d="M51.688 2c0 7.917-4.277 15-10.313 15c6.035 0 10.313 7.084 10.313 15c0-7.916 4.277-15 10.313-15c-6.036 0-10.313-7.083-10.313-15"></path>
                    </svg>
                </div>
                <h3>Welcome to Anna</h3>
                <p>Your intelligent CRM assistant is ready to help. Ask me anything about your customers, projects, or CRM data.</p>
                <div class="pc-welcome-suggestions">
                    <button onclick="insertDrawerMessage('Show me recent invoices')" class="pc-suggestion-chip">
                        💰 Recent Invoices
                    </button>
                    <button onclick="insertDrawerMessage('Find customer information')" class="pc-suggestion-chip">
                        👥 Customer Info
                    </button>
                    <button onclick="insertDrawerMessage('Project status updates')" class="pc-suggestion-chip">
                        📊 Project Status
                    </button>
                </div>
            </div>
            
            <!-- Chat Messages -->
            <div class="pc-drawer-messages" id="pc-widget-messages">
                <!-- Messages will be dynamically added here -->
            </div>
        </div>
        
        <!-- Chat Input -->
        <div class="pc-drawer-input">
            <div class="pc-input-container">
                <div class="pc-input-wrapper">
                    <!-- Ghost text for inline completion -->
                    <div class="pc-ghost-text" id="pc-widget-ghost-text"></div>
                    <textarea 
                        class="pc-textarea" 
                        id="pc-widget-input" 
                        rows="1" 
                        placeholder="Ask your AI assistant anything..." 
                        onkeypress="handleWidgetKeyPress(event)"
                        oninput="handleWidgetInput(event)"
                        onkeydown="handleWidgetKeyDown(event)"
                    ></textarea>
                </div>
                <button 
                    class="pc-send-btn" 
                    onclick="sendWidgetMessage()" 
                    id="pc-widget-send-btn"
                >
                    <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"></path>
                    </svg>
                </button>
            </div>
            
            <!-- Input Help Footer -->
            <div class="pc-widget-input-help">
                <span class="pc-widget-help-item">
                    <kbd>↑</kbd><kbd>↓</kbd> Previous messages
                </span>
                <span class="pc-widget-help-item">
                    <kbd>Shift</kbd>+<kbd>Enter</kbd> New line
                </span>
                <span class="pc-widget-help-item">
                    <kbd>Tab</kbd> Complete word
                </span>
            </div>
        </div>
    </div>
</div>

<style>
/* Anna Drawer Widget Styles */
.pc-anna-widget {
    position: fixed;
    top: 0;
    right: 0;
    bottom: 0;
    z-index: 9999;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    pointer-events: none;
}

.pc-anna-widget * {
    box-sizing: border-box;
}

/* Floating Action Button */
.pc-fab {
    position: fixed;
    bottom: 24px;
    right: 24px;
    width: 64px;
    height: 64px;
    background: #2563eb;
    border-radius: 50%;
    box-shadow: 0 8px 32px rgba(37, 99, 235, 0.4);
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    pointer-events: all;
    z-index: 10001;
}
/* Move FAB to the left of the widget when open */
.pc-anna-widget.drawer-open .pc-fab {
    right: 648px; /* 624px (drawer width) + 24px gap */
    transition: right 0.4s cubic-bezier(0.4, 0, 0.2, 1);
}

.pc-fab:hover {
    background: #1d4ed8;
    transform: translateY(-4px) scale(1.05);
    box-shadow: 0 12px 40px rgba(37, 99, 235, 0.5);
}

.pc-fab-icon {
    color: white;
    transition: transform 0.3s ease;
}

.pc-fab:hover .pc-fab-icon {
    transform: rotate(10deg) scale(1.1);
}

.pc-fab-notification {
    position: absolute;
    top: -2px;
    right: -2px;
    background: #ef4444;
    color: white;
    border-radius: 50%;
    width: 20px;
    height: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 11px;
    font-weight: 600;
    animation: pc-fab-bounce 0.5s ease-in-out;
}

@keyframes pc-fab-bounce {
    0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
    40% { transform: translateY(-8px); }
    60% { transform: translateY(-4px); }
}

/* Overlay */
.pc-drawer-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(4px);
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    pointer-events: none;
}

.pc-drawer-overlay.active {
    opacity: 1;
    visibility: visible;
    pointer-events: all;
}

/* Drawer */
.pc-drawer {
    position: fixed;
    top: 0;
    right: 0;
    bottom: 0;
    width: 624px;
    max-width: 100vw;
    background: #ffffff;
    box-shadow: -10px 0 50px rgba(0, 0, 0, 0.15);
    transform: translateX(100%);
    transition: transform 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    display: flex;
    flex-direction: column;
    border-left: 1px solid rgba(226, 232, 240, 0.8);
    pointer-events: all;
}

.pc-drawer.active {
    transform: translateX(0);
}

/* Drawer Header */
.pc-drawer-header {
    background: #2563eb;
    color: white;
    padding: 0;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    position: relative;
    overflow: hidden;
}

.pc-drawer-header-main {
    padding: 24px 28px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    position: relative;
    z-index: 1;
}

.pc-drawer-brand {
    display: flex;
    align-items: center;
    gap: 16px;
}

.pc-drawer-avatar {
    width: 44px;
    height: 44px;
    background: rgba(255, 255, 255, 0.15);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    border: 1px solid rgba(255, 255, 255, 0.2);
}

.pc-drawer-brand-info h2 {
    margin: 0;
    font-size: 20px;
    font-weight: 700;
    letter-spacing: -0.5px;
}

.pc-drawer-status {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-top: 4px;
    opacity: 0.9;
}

.pc-status-indicator {
    width: 8px;
    height: 8px;
    background: #10b981;
    border-radius: 50%;
    animation: pc-pulse 2s infinite;
}

@keyframes pc-pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.5; }
}

.pc-drawer-status span {
    font-size: 13px;
    font-weight: 500;
}

.pc-drawer-actions {
    display: flex;
    gap: 8px;
}

.pc-drawer-action-btn,
.pc-drawer-close {
    background: rgba(255, 255, 255, 0.15);
    border: 1px solid rgba(255, 255, 255, 0.2);
    color: white;
    border-radius: 10px;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s ease;
}

.pc-drawer-action-btn:hover,
.pc-drawer-close:hover {
    background: rgba(255, 255, 255, 0.25);
    transform: scale(1.05);
}

/* Session Message in Drawer */
.pc-drawer-session-bar {
    display: none;
    background: #fef3c7;
    border-top: 1px solid rgba(245, 158, 11, 0.3);
    padding: 16px 28px;
    align-items: center;
    gap: 12px;
    animation: pc-drawer-slideDown 0.3s ease-out;
}

.pc-drawer-session-bar .pc-session-content {
    display: flex;
    align-items: center;
    gap: 10px;
    flex: 1;
    justify-content: center;
}

.pc-drawer-session-bar .pc-session-content svg {
    color: #d97706;
    flex-shrink: 0;
}

.pc-drawer-session-bar #widget-session-message-text {
    color: #92400e;
    font-size: 14px;
    font-weight: 600;
    line-height: 1.4;
    text-align: center;
}

@keyframes pc-drawer-slideDown {
    from {
        opacity: 0;
        max-height: 0;
        padding-top: 0;
        padding-bottom: 0;
    }
    to {
        opacity: 1;
        max-height: 80px;
        padding-top: 16px;
        padding-bottom: 16px;
    }
}

/* Drawer Content */
.pc-drawer-content {
    flex: 1;
    display: flex;
    flex-direction: column;
    overflow: hidden;
}

/* Welcome Screen */
.pc-drawer-welcome {
    padding: 40px 32px;
    text-align: center;
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
}

.pc-welcome-icon {
    color: #2563eb;
    margin-bottom: 20px;
}

.pc-drawer-welcome h3 {
    margin: 0 0 12px 0;
    font-size: 24px;
    font-weight: 700;
    color: #1f2937;
    letter-spacing: -0.5px;
}

.pc-drawer-welcome p {
    margin: 0 0 24px 0;
    color: #6b7280;
    font-size: 15px;
    line-height: 1.5;
}

.pc-welcome-suggestions {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.pc-suggestion-chip {
    background: #ffffff;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    padding: 12px 16px;
    font-size: 14px;
    color: #374151;
    cursor: pointer;
    transition: all 0.2s ease;
    text-align: left;
    font-weight: 500;
}

.pc-suggestion-chip:hover {
    background: #2563eb;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.3);
}

/* Chat Messages */
.pc-drawer-messages {
    flex: 1;
    padding: 20px 24px;
    overflow-y: auto;
    display: flex;
    flex-direction: column;
}

.pc-drawer-messages::-webkit-scrollbar {
    width: 6px;
}

.pc-drawer-messages::-webkit-scrollbar-track {
    background: #f1f5f9;
    border-radius: 3px;
}

.pc-drawer-messages::-webkit-scrollbar-thumb {
    background: #cbd5e1;
    border-radius: 3px;
}

.pc-drawer-messages::-webkit-scrollbar-thumb:hover {
    background: #94a3b8;
}

/* Chat bubble alignment and style fixes */
.pc-widget-message {
    display: flex;
    align-items: flex-end;
    gap: 12px;
    animation: pc-message-fadeIn 0.4s ease-out;
    margin-bottom: 16px;
    max-width: 80%;
}
.pc-widget-message:last-child {
    margin-bottom: 0;
}
.pc-widget-user-message {
    align-self: flex-end;
    flex-direction: row-reverse;
    justify-content: flex-end;
}
.pc-widget-ai-message {
    align-self: flex-start;
    flex-direction: row;
    justify-content: flex-start;
}
.pc-widget-user-message .pc-widget-message-content {
    background: #2563eb;
    color: white;
    border: none;
    border-radius: 18px;
    border-top-right-radius: 4px;
    border-top-left-radius: 18px;
    text-align: right;
    margin-left: 0;
    margin-right: 0;
}
.pc-widget-ai-message .pc-widget-message-content {
    background: #ffffff;
    border: 1px solid #e2e8f0;
    color: #374151;
    border-radius: 18px;
    border-top-left-radius: 4px;
    text-align: left;
    margin-left: 0;
    margin-right: 0;
}
.pc-widget-avatar-small {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
    font-weight: 600;
    font-size: 12px;
    margin-top: 4px;
}
.pc-widget-ai-message .pc-widget-avatar-small {
    background: #2563eb;
    color: white;
}
.pc-widget-user-message .pc-widget-avatar-small {
    background: #10b981;
    color: white;
}
.pc-widget-message-content {
    flex: 1;
    max-width: 320px;
    padding: 12px 16px;
    font-size: 14px;
    line-height: 1.5;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.04);
    display: inline-block;
    word-wrap: break-word;
}
.pc-widget-message-text p {
    margin: 0 0 8px 0;
}
.pc-widget-message-text p:last-child {
    margin-bottom: 0;
}
.pc-widget-timestamp {
    font-size: 11px;
    color: #9ca3af;
    margin-top: 4px;
    text-align: left;
    display: block;
}
.pc-widget-user-message .pc-widget-timestamp {
    text-align: right;
}

/* Drawer Input */
.pc-drawer-input {
    background: #ffffff;
    border-top: 1px solid #e2e8f0;
    padding: 0;
}


.pc-input-container {
    padding: 20px 24px;
    display: flex;
    align-items: flex-end;
    gap: 12px;
    position: relative;
}

.pc-input-wrapper {
    flex: 1;
    position: relative;
    background: #f8fafc;
    border: 2px solid #e2e8f0;
    border-radius: 16px;
    transition: all 0.2s ease;
}

.pc-input-wrapper:focus-within {
    background: #ffffff;
    border-color: #2563eb;
    box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
}

.pc-textarea {
    width: 100%;
    min-height: 20px;
    max-height: 120px;
    padding: 14px 16px;
    border: none;
    border-radius: 14px;
    font-size: 14px;
    line-height: 1.5;
    color: #374151;
    background: transparent;
    resize: none;
    outline: none;
    font-family: inherit;
    position: relative;
    z-index: 2;
}

.pc-textarea::placeholder {
    color: #9ca3af;
}

.pc-ghost-text {
    position: absolute;
    top: 14px;
    left: 16px;
    right: 16px;
    color: #c7d2fe;
    font-size: 14px;
    line-height: 1.5;
    pointer-events: none;
    white-space: pre-wrap;
    word-wrap: break-word;
    z-index: 1;
    display: none;
    font-family: inherit;
    overflow: hidden;
}

.pc-send-btn {
    width: 44px;
    height: 44px;
    background: #2563eb;
    border: none;
    border-radius: 12px;
    color: white;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s ease;
    flex-shrink: 0;
    box-shadow: 0 2px 8px rgba(37, 99, 235, 0.3);
}

.pc-send-btn:hover {
    background: #1d4ed8;
    transform: translateY(-1px) scale(1.02);
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.4);
}

.pc-send-btn:disabled {
    background: #9ca3af;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

/* Input Help Footer */
.pc-widget-input-help {
    padding: 12px 24px 16px;
    background: #f8fafc;
    display: flex;
    justify-content: center;
    gap: 20px;
    font-size: 11px;
    color: #6b7280;
    border-top: 1px solid #f1f5f9;
}

.pc-widget-help-item {
    display: flex;
    align-items: center;
    gap: 4px;
}

.pc-widget-help-item kbd {
    background: #e2e8f0;
    border: 1px solid #cbd5e1;
    border-radius: 4px;
    padding: 2px 6px;
    font-size: 10px;
    font-family: inherit;
    color: #475569;
    box-shadow: 0 1px 0 #cbd5e1;
}

/* Loading animation for AI response */
.pc-loading-message {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: #f8fafc;
    border-radius: 16px;
    color: #6b7280;
    font-style: italic;
}

.pc-loading-dots {
    display: flex;
    gap: 4px;
}

.pc-loading-dot {
    width: 6px;
    height: 6px;
    background: #6b7280;
    border-radius: 50%;
    animation: pulse 1.4s ease-in-out infinite both;
}

.pc-loading-dot:nth-child(1) { animation-delay: -0.32s; }
.pc-loading-dot:nth-child(2) { animation-delay: -0.16s; }
.pc-loading-dot:nth-child(3) { animation-delay: 0s; }

@keyframes pulse {
    0%, 80%, 100% {
        opacity: 0.3;
        transform: scale(0.8);
    }
    40% {
        opacity: 1;
        transform: scale(1);
    }
}

/* Responsive Design */
@media (max-width: 768px) {
    .pc-drawer {
        width: 100vw;
        max-width: 100vw;
    }
    
    .pc-fab {
        bottom: 20px;
        right: 20px;
        width: 56px;
        height: 56px;
    }
    
    .pc-drawer-header-main {
        padding: 20px 20px;
    }
    
    .pc-drawer-welcome {
        padding: 32px 24px;
    }
    
    .pc-drawer-messages {
        padding: 16px 20px;
    }
    
    .pc-input-container {
        padding: 16px 20px;
    }
    
    .pc-widget-input-help {
        padding: 10px 20px 14px;
        gap: 16px;
    }
}

@media (max-width: 480px) {
    .pc-drawer-brand-info h2 {
        font-size: 18px;
    }
    
    .pc-drawer-welcome h3 {
        font-size: 20px;
    }
    
    .pc-welcome-suggestions {
        gap: 6px;
    }
    
    .pc-widget-help-item span {
        display: none;
    }
}
</style>

<script>
// Anna Drawer JavaScript
window.drawerIsOpen = false;

function openAnnaChatPage() {
    window.location.href = '<?php echo admin_url('anna/chat'); ?>';
}

// Toggle drawer function
function toggleAnnaDrawer() {
    const widget = document.getElementById('perfex-anna-widget');
    const drawer = document.getElementById('pc-drawer');
    const overlay = document.getElementById('pc-drawer-overlay');
    
    window.drawerIsOpen = !window.drawerIsOpen;
    
    if (window.drawerIsOpen) {
        drawer.classList.add('active');
        overlay.classList.add('active');
        document.body.style.overflow = 'hidden';
        if (widget) widget.classList.add('drawer-open');
        // Hide welcome screen if there are messages
        const messagesContainer = document.getElementById('pc-widget-messages');
        const welcomeScreen = document.getElementById('pc-drawer-welcome');
        if (messagesContainer.children.length > 0 && welcomeScreen) {
            welcomeScreen.style.display = 'none';
        }
    } else {
        drawer.classList.remove('active');
        overlay.classList.remove('active');
        document.body.style.overflow = '';
        if (widget) widget.classList.remove('drawer-open');
    }
}

// Create new chat function for drawer
function createDrawerNewChat() {    
    // Clear current conversation ID
    window.widgetCurrentConversationId = null;
    
    // Clear conversation ID from localStorage
    localStorage.removeItem('anna_conversation_id');
    
    // Clear conversation manager state
    if (window.perfexConversationManager) {
        window.perfexConversationManager.clearConversationHistory();
    }
    
    // Clear messages
    const messagesContainer = document.getElementById('pc-widget-messages');
    messagesContainer.innerHTML = '';
    
    // Show welcome screen
    const welcomeScreen = document.getElementById('pc-drawer-welcome');
    if (welcomeScreen) {
        welcomeScreen.style.display = 'block';
    }
    
    // Clear input and word suggestions
    const input = document.getElementById('pc-widget-input');
    if (input) {
        input.value = '';
    }
    
    // Clear any word suggestions
    if (window.annaSync) {
        window.annaSync.clearWordSuggestion();
    }
    
    // Hide any session messages
    hideWidgetSessionMessage();
    
}

// Insert suggestion message
function insertDrawerMessage(message) {
    const input = document.getElementById('pc-widget-input');
    if (input) {
        input.value = message;
        input.focus();
    }
}

// Legacy compatibility - map old function names to new ones
window.toggleAnnaWidget = toggleAnnaDrawer;
window.createWidgetNewChat = createDrawerNewChat;

// Update widget status function
function updateWidgetStatus(status) {
    const statusElement = document.getElementById('pc-drawer-status-text');
    if (statusElement) {
        statusElement.textContent = status;
    }
}

// Simple markdown parser for bold text
function parseMarkdown(text) {
    // Convert **text** to <strong>text</strong>
    return text.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
}

// Show loading message for AI response
function showLoadingMessage() {
    const messagesContainer = document.getElementById('pc-widget-messages');
    if (!messagesContainer) return null;
    
    const loadingDiv = document.createElement('div');
    loadingDiv.className = 'pc-widget-message pc-widget-ai-message';
    loadingDiv.id = 'pc-loading-message';
    loadingDiv.innerHTML = `
        <div class="pc-widget-avatar-small">
            <svg width="16" height="16" viewBox="0 0 64 64" fill="currentColor">
                <path d="M22.625 2c0 15.834-8.557 30-20.625 30c12.068 0 20.625 14.167 20.625 30c0-15.833 8.557-30 20.625-30c-12.068 0-20.625-14.166-20.625-30"></path>
                <path d="M47 32c0 7.918-4.277 15-10.313 15C42.723 47 47 54.084 47 62c0-7.916 4.277-15 10.313-15C51.277 47 47 39.918 47 32z"></path>
                <path d="M51.688 2c0 7.917-4.277 15-10.313 15c6.035 0 10.313 7.084 10.313 15c0-7.916 4.277-15 10.313-15c-6.036 0-10.313-7.083-10.313-15"></path>
            </svg>
        </div>
        <div class="pc-widget-message-content">
            <div class="pc-loading-message">
                AI is thinking
                <div class="pc-loading-dots">
                    <div class="pc-loading-dot"></div>
                    <div class="pc-loading-dot"></div>
                    <div class="pc-loading-dot"></div>
                </div>
            </div>
        </div>
    `;
    
    messagesContainer.appendChild(loadingDiv);
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
    
    return loadingDiv;
}

// Remove loading message
function hideLoadingMessage() {
    const loadingElement = document.getElementById('pc-loading-message');
    if (loadingElement) {
        loadingElement.remove();
    }
}

// Override addWidgetMessage to work with new structure
const originalAddWidgetMessage = window.addWidgetMessage;
window.addWidgetMessage = function(message, type, isError = false) {
    // Skip empty messages
    if (!message || message.trim() === '') {
        return;
    }
    
    // Hide welcome screen when first message is added
    const welcomeScreen = document.getElementById('pc-drawer-welcome');
    if (welcomeScreen) {
        welcomeScreen.style.display = 'none';
    }
    
    // Call original function if it exists
    if (originalAddWidgetMessage) {
        return originalAddWidgetMessage(message, type, isError);
    }
    
    // Fallback implementation
    const messagesContainer = document.getElementById('pc-widget-messages');
    if (!messagesContainer) return;
    
    const messageDiv = document.createElement('div');
    const timestamp = new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    
    // Get user initials dynamically
    const getUserInitials = () => {
        const fullName = '<?php echo get_staff_full_name(); ?>';
        const nameParts = fullName.trim().split(' ');
        if (nameParts.length >= 2) {
            return (nameParts[0].charAt(0) + nameParts[nameParts.length - 1].charAt(0)).toUpperCase();
        } else {
            return fullName.substring(0, 2).toUpperCase();
        }
    };
    
    // Process message content
    let processedMessage = message.replace(/\n/g, '<br>');
    
    // Apply markdown parsing for AI messages only (to preserve user input as-is)
    if (type === 'ai' || type === 'assistant') {
        processedMessage = parseMarkdown(processedMessage);
    }
    
    if (type === 'user') {
        messageDiv.className = 'pc-widget-message pc-widget-user-message';
        messageDiv.innerHTML = `
            <div class="pc-widget-avatar-small">${getUserInitials()}</div>
            <div class="pc-widget-message-content">
                <div class="pc-widget-message-text">${processedMessage}</div>
                <div class="pc-widget-timestamp">${timestamp}</div>
            </div>
        `;
    } else {
        messageDiv.className = 'pc-widget-message pc-widget-ai-message';
        messageDiv.innerHTML = `
            <div class="pc-widget-avatar-small">
                <svg width="16" height="16" viewBox="0 0 64 64" fill="currentColor">
                    <path d="M22.625 2c0 15.834-8.557 30-20.625 30c12.068 0 20.625 14.167 20.625 30c0-15.833 8.557-30 20.625-30c-12.068 0-20.625-14.166-20.625-30"></path>
                    <path d="M47 32c0 7.918-4.277 15-10.313 15C42.723 47 47 54.084 47 62c0-7.916 4.277-15 10.313-15C51.277 47 47 39.918 47 32z"></path>
                    <path d="M51.688 2c0 7.917-4.277 15-10.313 15c6.035 0 10.313 7.084 10.313 15c0-7.916 4.277-15 10.313-15c-6.036 0-10.313-7.083-10.313-15"></path>
                </svg>
            </div>
            <div class="pc-widget-message-content">
                <div class="pc-widget-message-text" style="${isError ? 'background: #fef2f2; color: #dc2626; border-color: #fecaca;' : ''}">${processedMessage}</div>
                <div class="pc-widget-timestamp">${timestamp}</div>
            </div>
        `;
    }
    
    messagesContainer.appendChild(messageDiv);
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
};

// Minimal AnnaSync class embedded
if (!window.AnnaSync) {
    window.AnnaSync = class {
        constructor() {
            this.messageHistory = [];
            this.historyIndex = -1;
            this.currentInput = '';
            this.isNavigating = false;
            this.storageKey = 'pc_message_history';
            this.maxHistorySize = 100;
            this.tabId = 'tab_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            
            // Word completion variables
            this.wordSuggestions = [];
            this.currentSuggestion = '';
            this.suggestionPosition = -1;
            
            // Common CRM-related words for autocompletion
            this.commonWords = [
                'customer', 'customers', 'invoice', 'invoices', 'project', 'projects', 
                'task', 'tasks', 'lead', 'leads', 'contact', 'contacts', 'payment', 'payments',
                'estimate', 'estimates', 'proposal', 'proposals', 'contract', 'contracts',
                'ticket', 'tickets', 'staff', 'dashboard', 'report', 'reports', 'analytics',
                'revenue', 'expense', 'expenses', 'client', 'clients', 'deadline', 'deadlines',
                'status', 'pending', 'completed', 'active', 'inactive', 'overdue', 'recurring'
            ];
            
            this.loadMessageHistory();
            this.setupStorageListener();
        }
        
        loadMessageHistory() {
            try {
                const stored = localStorage.getItem(this.storageKey);
                if (stored) {
                    this.messageHistory = JSON.parse(stored);
                }
            } catch (error) {
                this.messageHistory = [];
            }
        }
        
        saveMessageHistory() {
            try {
                localStorage.setItem(this.storageKey, JSON.stringify(this.messageHistory));
            } catch (error) {
            }
        }
        
        addMessageToHistory(message) {
            if (!message || message.trim() === '') return;
            const trimmedMessage = message.trim();
            const existingIndex = this.messageHistory.indexOf(trimmedMessage);
            if (existingIndex > -1) {
                this.messageHistory.splice(existingIndex, 1);
            }
            this.messageHistory.push(trimmedMessage);
            if (this.messageHistory.length > this.maxHistorySize) {
                this.messageHistory = this.messageHistory.slice(-this.maxHistorySize);
            }
            this.historyIndex = -1;
            this.currentInput = '';
            this.saveMessageHistory();
        }
        
        navigateHistory(direction, inputElement) {
            if (!inputElement || this.messageHistory.length === 0) {
                return;
            }
            
            if (!this.isNavigating) {
                this.currentInput = inputElement.value;
                this.isNavigating = true;
            }
            
            const oldIndex = this.historyIndex;
            
            if (direction === 'up') {
                if (this.historyIndex < this.messageHistory.length - 1) {
                    this.historyIndex++;
                }
            } else if (direction === 'down') {
                if (this.historyIndex > -1) {
                    this.historyIndex--;
                }
            }
            
            if (this.historyIndex === -1) {
                inputElement.value = this.currentInput;
                this.isNavigating = false;
            } else {
                const historyMessage = this.messageHistory[this.messageHistory.length - 1 - this.historyIndex];
                inputElement.value = historyMessage;
            }
            
            inputElement.setSelectionRange(inputElement.value.length, inputElement.value.length);
        }
        
        handleKeyDown(event) {
            if (event.key === 'ArrowUp') {
                event.preventDefault();
                this.navigateHistory('up', event.target);
            } else if (event.key === 'ArrowDown') {
                event.preventDefault();
                this.navigateHistory('down', event.target);
            } else if (event.key === 'Enter' && !event.shiftKey) {
                this.isNavigating = false;
                this.historyIndex = -1;
                this.currentInput = '';
                this.clearWordSuggestion();
            } else if (event.key === 'Tab') {
                event.preventDefault();
                this.handleTabCompletion(event.target);
            } else if (event.key === 'Escape') {
                this.clearWordSuggestion();
            }
        }
        
        handleInput(event) {
            // If user starts typing while navigating, exit navigation mode
            if (this.isNavigating) {
                this.isNavigating = false;
                this.historyIndex = -1;
                this.currentInput = event.target.value;
            }
            
            // Handle word completion suggestions
            this.updateWordSuggestion(event.target);
        }
        
        updateWordSuggestion(inputElement) {
            const value = inputElement.value;
            const cursorPosition = inputElement.selectionStart;
            
            // Find the current word being typed
            const beforeCursor = value.substring(0, cursorPosition);
            const wordMatch = beforeCursor.match(/\b(\w+)$/);
            
            if (wordMatch) {
                const currentWord = wordMatch[1].toLowerCase();
                const startPosition = beforeCursor.length - currentWord.length;
                
                // Find matching words from common words and message history
                const allWords = [...this.commonWords, ...this.getWordsFromHistory()];
                const uniqueWords = [...new Set(allWords)];
                
                const matches = uniqueWords.filter(word => 
                    word.toLowerCase().startsWith(currentWord) && 
                    word.toLowerCase() !== currentWord
                ).slice(0, 5); // Limit to 5 suggestions
                
                if (matches.length > 0) {
                    this.showWordSuggestion(inputElement, matches[0], currentWord, startPosition);
                } else {
                    this.clearWordSuggestion();
                }
            } else {
                this.clearWordSuggestion();
            }
        }
        
        getWordsFromHistory() {
            const words = [];
            this.messageHistory.forEach(message => {
                const messageWords = message.toLowerCase().match(/\b\w+\b/g) || [];
                words.push(...messageWords.filter(word => word.length > 3));
            });
            return words;
        }
        
        showWordSuggestion(inputElement, suggestion, currentWord, startPosition) {
            this.currentSuggestion = suggestion;
            this.suggestionPosition = startPosition;
            
            // Update ghost text
            const ghostElement = document.getElementById('pc-widget-ghost-text');
            if (ghostElement) {
                const value = inputElement.value;
                const completion = suggestion.substring(currentWord.length);
                const beforeSuggestion = value.substring(0, startPosition + currentWord.length);
                const afterSuggestion = value.substring(inputElement.selectionStart);
                
                ghostElement.textContent = beforeSuggestion + completion + afterSuggestion;
                ghostElement.style.display = 'block';
            }
        }
        
        clearWordSuggestion() {
            this.currentSuggestion = '';
            this.suggestionPosition = -1;
            
            const ghostElement = document.getElementById('pc-widget-ghost-text');
            if (ghostElement) {
                ghostElement.style.display = 'none';
                ghostElement.textContent = '';
            }
        }
        
        handleTabCompletion(inputElement) {
            if (this.currentSuggestion && this.suggestionPosition >= 0) {
                const value = inputElement.value;
                const cursorPosition = inputElement.selectionStart;
                const beforeCursor = value.substring(0, cursorPosition);
                const wordMatch = beforeCursor.match(/\b(\w+)$/);
                
                if (wordMatch) {
                    const currentWord = wordMatch[1];
                    const startPosition = this.suggestionPosition;
                    const endPosition = startPosition + currentWord.length;
                    
                    // Replace the current word with the suggestion
                    const newValue = value.substring(0, startPosition) + 
                                   this.currentSuggestion + 
                                   value.substring(endPosition);
                    
                    inputElement.value = newValue;
                    
                    // Set cursor position after the completed word
                    const newCursorPosition = startPosition + this.currentSuggestion.length;
                    inputElement.setSelectionRange(newCursorPosition, newCursorPosition);
                    
                    this.clearWordSuggestion();
                }
            }
        }
        
        setupMessageHistoryNavigation(inputSelector) {
            const inputElement = document.querySelector(inputSelector);
            if (!inputElement) {
                return;
            }
            
            // Remove existing listeners
            if (this.boundHandleKeyDown) {
                inputElement.removeEventListener('keydown', this.boundHandleKeyDown);
            }
            if (this.boundHandleInput) {
                inputElement.removeEventListener('input', this.boundHandleInput);
            }
            
            // Bind methods to preserve context
            this.boundHandleKeyDown = this.handleKeyDown.bind(this);
            this.boundHandleInput = this.handleInput.bind(this);
            
            // Add event listeners
            inputElement.addEventListener('keydown', this.boundHandleKeyDown);
            inputElement.addEventListener('input', this.boundHandleInput);
        }
        
        broadcastMessage(conversationId, message, response) {
            this.addMessageToHistory(message);
            const data = {
                type: 'new_message',
                conversation_id: conversationId,
                message: message,
                response: response,
                timestamp: Date.now(),
                tab_id: this.tabId
            };
            
            localStorage.setItem('pc_current_conversation_id', conversationId);
            const syncKey = 'pc_cross_tab_sync_' + Date.now();
            localStorage.setItem(syncKey, JSON.stringify(data));
        }
        
        setupStorageListener() {
            window.addEventListener('storage', (event) => {
                if (event.key && event.key.startsWith('pc_cross_tab_sync_') && event.newValue) {
                    try {
                        const data = JSON.parse(event.newValue);
                        if (data.tab_id !== this.tabId) {
                            this.handleCrossTabMessage(data);
                        }
                    } catch (error) {
                        console.error('Error parsing cross-tab message:', error);
                    }
                }
            });
        }
        
        handleCrossTabMessage(data) {
            if (data.type === 'new_message') {
                const welcomeScreen = document.getElementById('pc-drawer-welcome');
                if (welcomeScreen) {
                    welcomeScreen.style.display = 'none';
                }
                if (typeof addWidgetMessage === 'function') {
                    addWidgetMessage(data.message, 'user');
                    addWidgetMessage(data.response, 'ai');
                }
                
                // Store in conversation history as well
                if (window.perfexConversationManager) {
                    window.perfexConversationManager.addMessageToConversation(data.message, 'user');
                    window.perfexConversationManager.addMessageToConversation(data.response, 'assistant');
                }
                
                this.addMessageToHistory(data.message);
            }
        }
        
        getHistory() {
            return this.messageHistory;
        }
    };
}

// Minimal conversation manager embedded - uses DB via API instead of localStorage
if (!window.AnnaConversationManager) {
    window.AnnaConversationManager = class {
        constructor() {
            this.storageKey = 'anna_conversation_id';
            this.currentConversationId = null;
            this.conversationHistory = [];
            this.isInitialized = false;
        }
        
        initialize() {
            return new Promise((resolve) => {
                this.currentConversationId = localStorage.getItem(this.storageKey);
                
                if (this.currentConversationId) {
                    this.loadConversationFromDB(this.currentConversationId).then(() => {
                        this.isInitialized = true;
                        resolve();
                    }).catch((error) => {
                        this.conversationHistory = [];
                        this.isInitialized = true;
                        resolve();
                    });
                } else {
                    this.conversationHistory = [];
                    this.isInitialized = true;
                    resolve();
                }
            });
        }
        
        loadConversationFromDB(conversationId) {
            return new Promise((resolve, reject) => {
                const url = window.admin_url ? 
                    window.admin_url + 'anna/get_conversation/' + conversationId : 
                    'admin/anna/get_conversation/' + conversationId;
                
                fetch(url, {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json',
                    }
                })
                .then(response => response.json())
                .then(res => {
                    if (res.success && res.conversation && res.conversation.messages) {
                        // Convert DB messages to our internal format
                        this.conversationHistory = res.conversation.messages.map((msg, index) => ({
                            id: msg.id || (Date.now() + index), // Use provided ID or generate one
                            role: msg.role,
                            content: msg.content, // Content is already in the correct field
                            created_at: msg.created_at,
                            conversation_id: conversationId
                        }));
                        
                        resolve();
                    } else {
                        this.conversationHistory = [];
                        resolve();
                    }
                })
                .catch(error => {
                    reject(error);
                });
            });
        }
        
        addMessageToConversation(message, role) {
            const messageObj = {
                id: Date.now(),
                role: role,
                content: message,
                created_at: new Date().toISOString(),
                conversation_id: this.currentConversationId
            };
            
            this.conversationHistory.push(messageObj);
            return messageObj;
        }
        
        storeConversationId(conversationId) {
            this.currentConversationId = conversationId;
            localStorage.setItem(this.storageKey, conversationId);
        }
        
        clearConversationHistory() {
            this.conversationHistory = [];
            this.currentConversationId = null;
            localStorage.removeItem(this.storageKey);
        }
        
        static getInstance() {
            if (!window.perfexConversationManagerInstance) {
                window.perfexConversationManagerInstance = new window.AnnaConversationManager();
            }
            return window.perfexConversationManagerInstance;
        }
    };
}

// --- Initialization for advanced features ---
document.addEventListener('DOMContentLoaded', function() {
    // Ensure drawer starts closed
    window.drawerIsOpen = false;
    const drawer = document.getElementById('pc-drawer');
    const overlay = document.getElementById('pc-drawer-overlay');
    if (drawer) drawer.classList.remove('active');
    if (overlay) overlay.classList.remove('active');

    // Initialize features in sequence with delay to ensure all scripts loaded
    setTimeout(() => {
        initializeWidgetFeatures();
    }, 100);
});

// Initialize widget features
function initializeWidgetFeatures() {
    // 1. Initialize Conversation Manager and load most recent chat
    initializeConversationManager()
        .then(() => {
            // 2. Initialize sync features
            initializeSyncFeatures();
            
            // 3. Initialize real-time features
            initializeRealTimeFeatures();
            
        })
        .catch(error => {
        });
}

// Initialize conversation manager
function initializeConversationManager() {
    return new Promise((resolve, reject) => {
        if (window.AnnaConversationManager) {
            try {
                const conversationManager = window.AnnaConversationManager.getInstance ? 
                    window.AnnaConversationManager.getInstance() : 
                    new window.AnnaConversationManager();
                
                // Store reference globally
                window.perfexConversationManager = conversationManager;
                
                // Initialize and load conversation history
                if (conversationManager.initialize) {
                    conversationManager.initialize().then(() => {
                        loadConversationHistory(conversationManager);
                        window.widgetCurrentConversationId = conversationManager.currentConversationId;
                        resolve();
                    }).catch(error => {
                        // Continue anyway
                        loadConversationHistory(conversationManager);
                        window.widgetCurrentConversationId = conversationManager.currentConversationId;
                        resolve();
                    });
                } else {
                    loadConversationHistory(conversationManager);
                    window.widgetCurrentConversationId = conversationManager.currentConversationId;
                    resolve();
                }
            } catch (error) {
                resolve();
            }
        } else {
            resolve();
        }
    });
}

// Load conversation history into widget
function loadConversationHistory(conversationManager) {
    if (conversationManager.conversationHistory && conversationManager.conversationHistory.length > 0) {
        const messagesContainer = document.getElementById('pc-widget-messages');
        if (!messagesContainer) {
            return;
        }
        
        messagesContainer.innerHTML = '';
        
        conversationManager.conversationHistory.forEach((msg, index) => {
            if (msg.role === 'user') {
                addWidgetMessage(msg.content, 'user');
            } else if (msg.role === 'assistant' || msg.role === 'ai') {
                addWidgetMessage(msg.content, 'ai');
            }
        });
        
        // Hide welcome screen if there are messages
        const welcomeScreen = document.getElementById('pc-drawer-welcome');
        if (welcomeScreen) {            
            welcomeScreen.style.display = 'none';
        }
        
        // Scroll to bottom of messages
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
        
    } else {
        // Ensure welcome screen is visible
        const welcomeScreen = document.getElementById('pc-drawer-welcome');
        if (welcomeScreen) {
            welcomeScreen.style.display = 'block';
        }
    }
}

// Initialize sync features
function initializeSyncFeatures() {
    if (window.AnnaSync) {
        window.annaSync = new window.AnnaSync();
        
        // Set up message history navigation
        const inputElement = document.getElementById('pc-widget-input');
        if (inputElement) {
            window.annaSync.setupMessageHistoryNavigation('#pc-widget-input');
            
            // Test if event handlers are working
        } else {
        }
        
        // Check for existing conversation ID in localStorage
        const existingConversationId = localStorage.getItem('pc_current_conversation_id');
        if (existingConversationId && existingConversationId !== window.widgetCurrentConversationId) {
            window.widgetCurrentConversationId = existingConversationId;
            window.annaSync.loadConversationFromSync(existingConversationId);
        }
    }
}

// Initialize real-time features
function initializeRealTimeFeatures() {
    if (window.AnnaRealTime) {
        window.annaRealTime = new window.AnnaRealTime();
        // Set up event listeners for real-time updates
        window.annaRealTime.on('new_message', (data) => {
            if (data.conversation_id === window.widgetCurrentConversationId) {
                addWidgetMessage(data.message, 'user');
                addWidgetMessage(data.response, 'ai');
            }
        });
    } else {
    }
}

// Patch AnnaSync for extra debug logging
if (window.AnnaSync) {
    const origBroadcastMessage = window.AnnaSync.prototype.broadcastMessage;
    window.AnnaSync.prototype.broadcastMessage = function(conversationId, message, response) {
        return origBroadcastMessage.call(this, conversationId, message, response);
    };
    const origSyncNewMessage = window.AnnaSync.prototype.syncNewMessage;
    window.AnnaSync.prototype.syncNewMessage = function(data) {
        return origSyncNewMessage.call(this, data);
    };
}

// Session message functions (override with new structure)
function showWidgetSessionMessage(message) {
    const container = document.getElementById('widget-session-message-container');
    const textElement = document.getElementById('widget-session-message-text');
    
    if (container && textElement) {
        textElement.textContent = message;
        container.style.display = 'flex';
    }
}

function hideWidgetSessionMessage() {
    const container = document.getElementById('widget-session-message-container');
    if (container) {
        container.style.display = 'none';
    }
}

// --- Add sendWidgetMessage implementation ---
function sendWidgetMessage() {
    const input = document.getElementById('pc-widget-input');
    const message = input.value.trim();
    const sendBtn = document.getElementById('pc-widget-send-btn');

    if (!message) return;

    // Add user message to chat
    addWidgetMessage(message, 'user');

    // Show loading message for AI response
    showLoadingMessage();

    // Clear input and disable while sending
    input.value = '';
    input.disabled = true;
    sendBtn.disabled = true;
    sendBtn.innerHTML = `<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24" style="animation: spin 1s linear infinite;"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path></svg>`;

    // Prepare data for AJAX
    const data = {
        message: message,
        conversation_id: window.widgetCurrentConversationId || '',
        auth_key: (window.app && window.app.anna && window.app.anna.loggedUser && window.app.anna.loggedUser.auth_key) || '',
        key: (window.app && window.app.anna && window.app.anna.loggedUser && window.app.anna.loggedUser.key) || '',
        csrf_token_name: '<?php echo $this->security->get_csrf_hash(); ?>'
    };
    

    // Send AJAX request
    fetch(window.admin_url ? window.admin_url + 'anna/process_chat' : 'admin/anna/process_chat', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: new URLSearchParams(data)
    })
    .then(response => response.json())
    .then(res => {
        // Hide loading message before showing response
        hideLoadingMessage();
        
        if (res.success && res.response) {
            addWidgetMessage(res.response, 'ai');
            
            // Store messages in conversation history
            if (window.perfexConversationManager) {
                window.perfexConversationManager.addMessageToConversation(message, 'user');
                window.perfexConversationManager.addMessageToConversation(res.response, 'assistant');
            }
            
            // Add to message history for autocomplete
            if (window.annaSync) {
                window.annaSync.addMessageToHistory(message);
                window.annaSync.broadcastMessage(window.widgetCurrentConversationId || res.conversation_id, message, res.response);
            }
        } else if (res.error) {
            addWidgetMessage('Error: ' + res.error, 'ai', true);
        } else {
            addWidgetMessage('I apologize, but I encountered an error. Please try again.', 'ai', true);
        }
        // Store conversation ID if returned
        if (res.conversation_id) {
            window.widgetCurrentConversationId = res.conversation_id;
            // Update conversation manager
            if (window.perfexConversationManager) {
                window.perfexConversationManager.storeConversationId(res.conversation_id);
            }
            // Sync conversation ID to other tabs
            if (window.annaSync) {
                localStorage.setItem('pc_current_conversation_id', res.conversation_id);
            }
        }
        // Show session message if present
        if (res.session_message) {
            showWidgetSessionMessage(res.session_message);
        }
    })
    .catch(() => {
        // Hide loading message on error
        hideLoadingMessage();
        addWidgetMessage('Unable to connect to the AI service. Please check your connection and try again.', 'ai', true);
    })
    .finally(() => {
        input.disabled = false;
        sendBtn.disabled = false;
        sendBtn.innerHTML = `<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"></path></svg>`;
        input.focus();
    });
}

// Handle Enter key for sending
function handleWidgetKeyPress(event) {
    if (event.key === 'Enter' && !event.shiftKey) {
        event.preventDefault();
        sendWidgetMessage();
    }
}

// --- Add missing input handlers for widget input ---
function handleWidgetKeyDown(event) {
    // If AnnaSync is available, let it handle message history navigation
    if (window.annaSync && typeof window.annaSync.handleKeyDown === 'function') {
        window.annaSync.handleKeyDown(event);
    }
    // You can add more logic here for autocomplete, etc.
}

function handleWidgetInput(event) {
    // If AnnaSync is available, let it handle input for autocomplete, etc.
    if (window.annaSync && typeof window.annaSync.handleInput === 'function') {
        window.annaSync.handleInput(event);
    }
    // You can add more logic here for inline completion, etc.
}
</script>