<?php
// modules/anna/views/chat_with_sidebar.php
defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<input type="hidden" name="<?php echo $this->security->get_csrf_token_name(); ?>" value="<?php echo $this->security->get_csrf_hash(); ?>">
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <!-- ChatGPT-like Layout -->
                <div class="pc-chat-layout">
                    <!-- Sidebar -->
                    <div class="pc-sidebar">
                        <!-- Sidebar Header -->
                        <div class="pc-sidebar-header">
                            <button class="pc-new-chat-btn" onclick="createNewChat()">
                                <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                                </svg>
                                New Chat
                            </button>
                        </div>

                        <!-- Conversations List -->
                        <div class="pc-conversations-list" id="conversations-list">
                            <?php if (isset($conversations) && !empty($conversations)): ?>
                                <?php foreach ($conversations as $conv): ?>
                                    <div class="pc-conversation-item <?php echo ($conversation_id == $conv['id']) ? 'active' : ''; ?>" 
                                         data-conversation-id="<?php echo $conv['id']; ?>"
                                         onclick="switchConversation(<?php echo $conv['id']; ?>)">
                                        <div class="pc-conversation-content">
                                            <div class="pc-conversation-title" data-original-title="<?php echo htmlspecialchars($conv['title']); ?>">
                                                <?php echo htmlspecialchars($conv['title']); ?>
                                            </div>
                                            <div class="pc-conversation-meta">
                                                <span class="pc-message-count"><?php echo $conv['message_count']; ?> messages</span>
                                                <span class="pc-last-activity"><?php echo time_ago($conv['last_activity']); ?></span>
                                            </div>
                                        </div>
                                        <div class="pc-conversation-actions">
                                            <button onclick="event.stopPropagation(); renameConversation(<?php echo $conv['id']; ?>)" class="pc-action-btn" title="Rename">
                                                <svg width="12" height="12" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                                                </svg>
                                            </button>
                                            <button onclick="event.stopPropagation(); deleteConversation(<?php echo $conv['id']; ?>)" class="pc-action-btn" title="Delete">
                                                <svg width="12" height="12" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                                                </svg>
                                            </button>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="pc-no-conversations">
                                    <p>No conversations yet. Start a new chat!</p>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Sidebar Footer -->
                        <div class="pc-sidebar-footer">
                            <div class="pc-user-info">
                                <div class="pc-user-avatar">
                                    <?php echo strtoupper(substr(get_staff_full_name(), 0, 1)); ?>
                                </div>
                                <div class="pc-user-details">
                                    <span class="pc-user-name"><?php echo get_staff_full_name(); ?></span>
                                    <span class="pc-user-role">Support Agent</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Main Chat Area -->
                    <div class="pc-main-chat">
                        <!-- Chat Header -->
                        <div class="pc-chat-header">
                            <div class="pc-chat-title">
                                <h4 id="current-chat-title">
                                    <?php if (isset($current_conversation)): ?>
                                        <?php echo htmlspecialchars($current_conversation['conversation']['title']); ?>
                                    <?php else: ?>
                                        Anna
                                    <?php endif; ?>
                                </h4>
                                <span class="pc-chat-status">
                                    <div class="pc-status-dot"></div>
                                    AI Assistant - Online
                                </span>
                            </div>
                            <div class="pc-chat-actions">
                                <button type="button" class="btn btn-default btn-sm" onclick="exportChat()">
                                    <i class="fa fa-download"></i> Export
                                </button>
                                <button type="button" class="btn btn-default btn-sm" onclick="clearCurrentChat()">
                                    <i class="fa fa-refresh"></i> Clear
                                </button>
                            </div>
                            
                            <!-- Session Message as Header Extension -->
                            <div id="session-message-container" class="pc-session-header-bar" style="display: none;">
                                <div class="pc-session-content">
                                    <svg width="16" height="16" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"></path>
                                    </svg>
                                    <span id="session-message-text"></span>
                                </div>
                            </div>
                        </div>

                        <!-- Messages Container -->
                        <div class="pc-messages-container">
                            <div id="chat-messages" class="pc-messages">
                                <?php if (isset($current_conversation) && !empty($current_conversation['messages'])): ?>
                                    <?php foreach ($current_conversation['messages'] as $msg): ?>
                                        <div class="pc-message <?php echo $msg['role'] === 'user' ? 'pc-message-user' : 'pc-message-ai'; ?>">
                                            <?php if ($msg['role'] === 'user'): ?>
                                                <div class="pc-avatar">
                                                    <?php echo strtoupper(substr(get_staff_full_name(), 0, 1) . (strpos(get_staff_full_name(), ' ') ? substr(get_staff_full_name(), strpos(get_staff_full_name(), ' ') + 1, 1) : '')); ?>
                                                </div>
                                                <div class="pc-message-content">
                                                    <div class="pc-message-text"><?php echo nl2br(htmlspecialchars($msg['content'])); ?></div>
                                                    <span class="pc-timestamp"><?php echo date('H:i', strtotime($msg['created_at'])); ?></span>
                                                </div>
                                            <?php else: ?>
                                                <div class="pc-avatar">
                                                    <svg width="20" height="20" fill="currentColor" viewBox="0 0 20 20">
                                                        <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                                    </svg>
                                                </div>
                                                <div class="pc-message-content">
                                                    <div class="pc-message-text"><?php echo nl2br(htmlspecialchars($msg['content'])); ?></div>
                                                    <span class="pc-timestamp"><?php echo date('H:i', strtotime($msg['created_at'])); ?></span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <!-- Welcome Message -->
                                    <div class="pc-message pc-message-ai">
                                        <div class="pc-avatar">
                                            <svg width="20" height="20" fill="currentColor" viewBox="0 0 20 20">
                                                <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                            </svg>
                                        </div>
                                        <div class="pc-message-content">
                                            <div class="pc-message-text">
                                                <p><strong>👋 Hello <?php echo get_staff_full_name(); ?>!</strong></p>
                                                <p>I'm your AI assistant for Perfex CRM. I can help you with:</p>
                                                <div class="pc-capabilities">
                                                    <div class="pc-capability">
                                                        <span class="pc-bullet">•</span>
                                                        <span>Finding customers and their information</span>
                                                    </div>
                                                    <div class="pc-capability">
                                                        <span class="pc-bullet">•</span>
                                                        <span>Managing invoices and payments</span>
                                                    </div>
                                                    <div class="pc-capability">
                                                        <span class="pc-bullet">•</span>
                                                        <span>Tracking leads and opportunities</span>
                                                    </div>
                                                    <div class="pc-capability">
                                                        <span class="pc-bullet">•</span>
                                                        <span>Generating reports and analytics</span>
                                                    </div>
                                                </div>
                                                <p class="pc-help-text">How can I assist you today?</p>
                                            </div>
                                            <span class="pc-timestamp"><?php echo date('H:i'); ?></span>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Input Area -->
                        <div class="pc-input-area">
                            <div class="pc-input-container">
                                <div class="pc-input-wrapper">
                                    <textarea 
                                        class="pc-textarea" 
                                        id="chat-input" 
                                        rows="1" 
                                        placeholder="Type your message here..." 
                                        onkeypress="handleKeyPress(event)"
                                    ></textarea>
                                </div>
                                <button 
                                    type="button" 
                                    class="pc-send-btn" 
                                    onclick="sendMessage()" 
                                    id="send-btn"
                                >
                                    <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"></path>
                                    </svg>
                                    <span>Send</span>
                                </button>
                            </div>
                            
                            <!-- Quick Actions -->
                            <div class="pc-quick-actions">
                                <div class="pc-quick-left">
                                    <button onclick="insertQuickMessage('Search for a customer')" class="pc-quick-btn">
                                        🔍 Search Customer
                                    </button>
                                    <button onclick="insertQuickMessage('Show me recent invoices')" class="pc-quick-btn">
                                        📄 Recent Invoices
                                    </button>
                                    <button onclick="insertQuickMessage('Generate a sales report')" class="pc-quick-btn">
                                        📊 Sales Report
                                    </button>
                                </div>
                                <div class="pc-help-hint">
                                    Press Enter to send • Shift+Enter for new line
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* ChatGPT-like Layout */
.pc-chat-layout {
    display: flex;
    height: calc(100vh - 200px);
    min-height: 600px;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    overflow: hidden;
    background: white;
}

/* Sidebar Styles */
.pc-sidebar {
    width: 300px;
    background: #f8fafc;
    border-right: 1px solid #e2e8f0;
    display: flex;
    flex-direction: column;
}

.pc-sidebar-header {
    padding: 16px;
    border-bottom: 1px solid #e2e8f0;
}

.pc-new-chat-btn {
    width: 100%;
    background: #2563eb;
    color: white;
    border: none;
    padding: 12px 16px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 500;
    cursor: pointer;
    transition: background-color 0.2s ease;
}

.pc-new-chat-btn:hover {
    background: #1d4ed8;
}

.pc-conversations-list {
    flex: 1;
    overflow-y: auto;
    padding: 8px;
}

.pc-conversation-item {
    display: flex;
    align-items: center;
    padding: 12px;
    margin: 4px 0;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
    background: transparent;
}

.pc-conversation-item:hover {
    background: #e2e8f0;
}

.pc-conversation-item.active {
    background: #dbeafe;
    border-left: 3px solid #2563eb;
}

.pc-conversation-content {
    flex: 1;
    min-width: 0;
}

.pc-conversation-title {
    font-weight: 500;
    font-size: 14px;
    color: #1f2937;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    margin-bottom: 4px;
}

.pc-conversation-meta {
    display: flex;
    justify-content: space-between;
    font-size: 12px;
    color: #6b7280;
}

.pc-conversation-actions {
    display: flex;
    gap: 4px;
    opacity: 0;
    transition: opacity 0.2s ease;
}

.pc-conversation-item:hover .pc-conversation-actions {
    opacity: 1;
}

.pc-action-btn {
    background: none;
    border: none;
    padding: 4px;
    border-radius: 4px;
    cursor: pointer;
    color: #6b7280;
    transition: all 0.2s ease;
}

.pc-action-btn:hover {
    background: #374151;
    color: white;
}

.pc-sidebar-footer {
    padding: 16px;
    border-top: 1px solid #e2e8f0;
}

.pc-user-info {
    display: flex;
    align-items: center;
    gap: 12px;
}

.pc-user-avatar {
    width: 32px;
    height: 32px;
    background: #2563eb;
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
    font-size: 14px;
}

.pc-user-details {
    display: flex;
    flex-direction: column;
}

.pc-user-name {
    font-weight: 500;
    font-size: 14px;
    color: #1f2937;
}

.pc-user-role {
    font-size: 12px;
    color: #6b7280;
}

/* Main Chat Area */
.pc-main-chat {
    flex: 1;
    display: flex;
    flex-direction: column;
}

.pc-chat-header {
    padding: 16px 24px;
    border-bottom: 1px solid #e2e8f0;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.pc-chat-title h4 {
    margin: 0 0 4px 0;
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
}

.pc-chat-status {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 14px;
    color: #6b7280;
}

.pc-status-dot {
    width: 8px;
    height: 8px;
    background: #10b981;
    border-radius: 50%;
    animation: pc-pulse 2s infinite;
}

.pc-messages-container {
    flex: 1;
    overflow: hidden;
    display: flex;
    flex-direction: column;
}

.pc-messages {
    flex: 1;
    padding: 24px;
    overflow-y: auto;
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.pc-message {
    display: flex;
    align-items: flex-start;
    gap: 12px;
    animation: pc-fadeInUp 0.4s ease-out;
}

.pc-message-user {
    justify-content: flex-end;
}

.pc-message-user .pc-message-content {
    order: -1;
}

.pc-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
}

.pc-message-ai .pc-avatar {
    background-color: #dbeafe;
    color: #2563eb;
}

.pc-message-user .pc-avatar {
    background-color: #2563eb;
    color: white;
    font-size: 14px;
    font-weight: 600;
}

.pc-message-content {
    max-width: 70%;
    background: white;
    border-radius: 16px;
    padding: 16px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.1);
    position: relative;
}

.pc-message-ai .pc-message-content {
    border-top-left-radius: 4px;
}

.pc-message-user .pc-message-content {
    background-color: #2563eb;
    color: white;
    border-top-right-radius: 4px;
}

.pc-message-text {
    line-height: 1.5;
}

.pc-message-text p {
    margin: 0 0 8px 0;
}

.pc-message-text p:last-child {
    margin-bottom: 0;
}

.pc-capabilities {
    margin: 12px 0;
}

.pc-capability {
    display: flex;
    align-items: flex-start;
    gap: 8px;
    margin: 8px 0;
}

.pc-bullet {
    color: #2563eb;
    font-weight: bold;
    margin-top: 2px;
}

.pc-help-text {
    color: #6b7280;
    font-size: 14px;
    margin-top: 12px !important;
}

.pc-timestamp {
    display: block;
    font-size: 12px;
    color: #9ca3af;
    margin-top: 8px;
}

.pc-message-user .pc-timestamp {
    color: rgba(255,255,255,0.7);
}

.pc-input-area {
    padding: 16px 24px;
    background: white;
    border-top: 1px solid #e2e8f0;
}

.pc-input-container {
    display: flex;
    align-items: flex-end;
    gap: 12px;
}

.pc-input-wrapper {
    flex: 1;
}

.pc-textarea {
    width: 100%;
    min-height: 44px;
    max-height: 120px;
    resize: none;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    padding: 12px 16px;
    font-size: 14px;
    line-height: 1.5;
    transition: all 0.2s ease;
    font-family: inherit;
}

.pc-textarea:focus {
    outline: none;
    border-color: #2563eb;
    box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
}

.pc-send-btn {
    background-color: #2563eb;
    color: white;
    border: none;
    border-radius: 8px;
    padding: 12px 24px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    gap: 8px;
    min-height: 44px;
}

.pc-send-btn:hover {
    background-color: #1d4ed8;
}

.pc-send-btn:disabled {
    background-color: #9ca3af;
    cursor: not-allowed;
}

.pc-quick-actions {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-top: 12px;
}

.pc-quick-left {
    display: flex;
    gap: 16px;
}

.pc-quick-btn {
    background: none;
    border: none;
    color: #6b7280;
    font-size: 12px;
    cursor: pointer;
    transition: color 0.2s ease;
    padding: 4px 0;
}

.pc-quick-btn:hover {
    color: #2563eb;
}

.pc-help-hint {
    font-size: 12px;
    color: #9ca3af;
}

/* Animations */
@keyframes pc-pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.5; }
}

@keyframes pc-fadeInUp {
    from {
        opacity: 0;
        transform: translateY(15px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Scrollbar Styles */
.pc-messages::-webkit-scrollbar,
.pc-conversations-list::-webkit-scrollbar {
    width: 6px;
}

.pc-messages::-webkit-scrollbar-track,
.pc-conversations-list::-webkit-scrollbar-track {
    background: #f1f5f9;
    border-radius: 3px;
}

.pc-messages::-webkit-scrollbar-thumb,
.pc-conversations-list::-webkit-scrollbar-thumb {
    background: #cbd5e1;
    border-radius: 3px;
}

.pc-messages::-webkit-scrollbar-thumb:hover,
.pc-conversations-list::-webkit-scrollbar-thumb:hover {
    background: #94a3b8;
}

/* No Conversations */
.pc-no-conversations {
    text-align: center;
    padding: 40px 20px;
    color: #6b7280;
}

/* Responsive */
@media (max-width: 968px) {
    .pc-sidebar {
        width: 250px;
    }
}

@media (max-width: 768px) {
    .pc-chat-layout {
        flex-direction: column;
        height: auto;
        min-height: 500px;
    }
    
    .pc-sidebar {
        width: 100%;
        max-height: 200px;
    }
    
    .pc-message-content {
        max-width: 85%;
    }
    
    .pc-quick-left {
        flex-direction: column;
        gap: 8px;
    }
    
    .pc-help-hint {
        display: none;
    }
}

/* Session Message Header Bar Styles */
.pc-session-header-bar {
    display: none;
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    border-top: 1px solid rgba(245, 158, 11, 0.3);
    padding: 12px 20px;
    margin: 0;
    align-items: center;
    gap: 12px;
    width: 100%;
    border-radius: 0;
    animation: slideDownHeader 0.3s ease-out;
    position: relative;
}

.pc-session-header-bar .pc-session-content {
    display: flex;
    align-items: center;
    gap: 10px;
    flex: 1;
    justify-content: center;
}

.pc-session-header-bar .pc-session-content svg {
    color: #d97706;
    flex-shrink: 0;
}

.pc-session-header-bar #session-message-text {
    color: #92400e;
    font-size: 14px;
    font-weight: 600;
    line-height: 1.4;
    text-align: center;
}

@keyframes slideDownHeader {
    from {
        opacity: 0;
        max-height: 0;
        padding-top: 0;
        padding-bottom: 0;
    }
    to {
        opacity: 1;
        max-height: 60px;
        padding-top: 12px;
        padding-bottom: 12px;
    }
}
</style>

<script>
// Global variables
let currentConversationId = <?php echo $conversation_id ?: 'null'; ?>;
let messageCount = 0;
let isTyping = false;

document.addEventListener('DOMContentLoaded', function() {
    initializeChat();
    initializeFullScreenChat();
});

// Initialize conversation manager for full-screen chat
async function initializeFullScreenChat() {
    try {
        const conversationManager = window.AnnaConversationManager.getInstance();
        await conversationManager.initialize();
        
        // If we have a conversation ID from the URL, use it
        const urlConversationId = <?php echo json_encode($conversation_id); ?>;
        if (urlConversationId && urlConversationId !== conversationManager.getCurrentConversationId()) {
            await conversationManager.loadConversation(urlConversationId);
            
            // Load conversation history into the chat
            const history = conversationManager.getConversationHistory();
            const messagesContainer = document.getElementById('chat-messages');
            
            if (messagesContainer && history.length > 0) {
                // Clear existing messages
                messagesContainer.innerHTML = '';
                
                // Add each message from history
                history.forEach(message => {
                    if (message.role === 'user') {
                        addMessageToChat(message.content, 'user');
                    } else if (message.role === 'assistant') {
                        addMessageToChat(message.content, 'ai');
                    }
                });
            }
        }
        
        // Update global conversation ID for compatibility
        currentConversationId = conversationManager.getCurrentConversationId();
    } catch (error) {
        // Use fallback conversation ID from URL if available
        const urlConversationId = <?php echo json_encode($conversation_id); ?>;
        if (urlConversationId) {
            currentConversationId = urlConversationId;
        }
    }
}

function initializeChat() {
    const input = document.getElementById('chat-input');
    
    // Auto-resize textarea
    input.addEventListener('input', function() {
        this.style.height = 'auto';
        this.style.height = Math.min(this.scrollHeight, 120) + 'px';
    });
    
    input.focus();
}

function createNewChat() {
    fetch('<?php echo admin_url('anna/new_conversation'); ?>', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: '<?php echo $this->security->get_csrf_token_name(); ?>=' + encodeURIComponent(getCsrfToken())
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            window.location.href = '<?php echo admin_url('anna/chat/'); ?>' + data.conversation_id;
        } else {
            alert('Failed to create new conversation');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error creating new conversation');
    });
}

function switchConversation(conversationId) {
    window.location.href = '<?php echo admin_url('anna/chat/'); ?>' + conversationId;
}

async function sendMessage() {
    const input = document.getElementById('chat-input');
    const message = input.value.trim();
    
    if (!message || isTyping) return;

    // Ensure conversation manager is available
    let conversationManager = null;
    if (window.AnnaConversationManager) {
        conversationManager = window.AnnaConversationManager.getInstance();
        if (!conversationManager.isInitialized) {
            await conversationManager.initialize();
        }
    }
    
    // Add user message
    addMessageToChat(message, 'user');
    
    // Clear input and show loading state
    input.value = '';
    input.style.height = 'auto';
    setLoadingState(true);
    showTypingIndicator();
     
    fetch('<?php echo admin_url('anna/process_chat'); ?>', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: new URLSearchParams({
            message: message,
            conversation_id: conversationManager ? conversationManager.getCurrentConversationId() || '' : currentConversationId || '',
            auth_key: app.anna.loggedUser.auth_key,
            key: app.anna.loggedUser.key,
            csrf_token_name: $('input[name=csrf_token_name]').val()
        })
    })
    .then(response => response.json())
    .then(data => {
        hideTypingIndicator();
        
        // Update conversation ID if returned
        if (data.conversation_id) {
            if (conversationManager) {
                conversationManager.storeConversationId(data.conversation_id);
            }
            currentConversationId = data.conversation_id;
        }
        
        // Handle session message if present
        if (data.session_message) {
            showSessionMessage(data.session_message);
        }
        
        // Handle response if present
        if (data.response) {
            // Store both user message and AI response in conversation history
            if (conversationManager) {
                conversationManager.addMessageToHistory(message, 'user');
                conversationManager.addMessageToHistory(data.response, 'assistant');
            }
            
            // Check if the response contains a confirmation request
            try {
                const parsedResponse = JSON.parse(data.response);
                if (parsedResponse.requires_confirmation) {
                    displayConfirmationRequest(parsedResponse);
                } else {
                    addMessageToChat(data.response, 'ai');
                }
            } catch (e) {
                // Response is not JSON, treat as normal message
                addMessageToChat(data.response, 'ai');
            }
            updateSidebarAfterMessage();
        } else if (!data.session_message) {
            // Only show error if there's no session message either
            addMessageToChat('I apologize, but I encountered an error. Please try again.', 'ai', true);
        }
    })
    .catch(error => {
        hideTypingIndicator();
        addMessageToChat('Unable to connect to the AI service. Please check your connection and try again.', 'ai', true);
    })
    .finally(() => {
        setLoadingState(false);
        input.focus();
    });
}

function addMessageToChat(message, type, isError = false) {
    const chatMessages = document.getElementById('chat-messages');
    const messageDiv = document.createElement('div');
    
    const currentTime = getCurrentTime();
    const userInitials = '<?php echo strtoupper(substr(get_staff_full_name(), 0, 1) . (strpos(get_staff_full_name(), " ") ? substr(get_staff_full_name(), strpos(get_staff_full_name(), " ") + 1, 1) : "")); ?>';
    
    if (type === 'user') {
        messageDiv.className = 'pc-message pc-message-user';
        messageDiv.innerHTML = `
            <div class="pc-avatar">${userInitials}</div>
            <div class="pc-message-content">
                <div class="pc-message-text">${escapeHtml(message)}</div>
                <span class="pc-timestamp">${currentTime}</span>
            </div>
        `;
    } else {
        const extraClass = isError ? ' style="background: #fef2f2; color: #dc2626;"' : '';
        messageDiv.className = 'pc-message pc-message-ai';
        messageDiv.innerHTML = `
            <div class="pc-avatar">
                <svg width="20" height="20" fill="currentColor" viewBox="0 0 20 20">
                    <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                </svg>
            </div>
            <div class="pc-message-content"${extraClass}>
                <div class="pc-message-text">${formatMessage(message)}</div>
                <span class="pc-timestamp">${currentTime}</span>
            </div>
        `;
    }
    
    chatMessages.appendChild(messageDiv);
    chatMessages.scrollTop = chatMessages.scrollHeight;
    messageCount++;
}

function showTypingIndicator() {
    const chatMessages = document.getElementById('chat-messages');
    const typingDiv = document.createElement('div');
    typingDiv.id = 'typing-indicator-container';
    typingDiv.className = 'pc-message pc-message-ai';
    typingDiv.innerHTML = `
        <div class="pc-avatar">
            <svg width="20" height="20" fill="currentColor" viewBox="0 0 20 20">
                <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
        </div>
        <div class="pc-typing-indicator" style="display: flex; align-items: center; padding: 12px 16px; background: white; border-radius: 16px; border-top-left-radius: 4px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="width: 8px; height: 8px; background-color: #cbd5e1; border-radius: 50%; animation: typing 1.4s infinite ease-in-out; margin-right: 4px;"></div>
            <div style="width: 8px; height: 8px; background-color: #cbd5e1; border-radius: 50%; animation: typing 1.4s infinite ease-in-out; animation-delay: -0.16s; margin-right: 4px;"></div>
            <div style="width: 8px; height: 8px; background-color: #cbd5e1; border-radius: 50%; animation: typing 1.4s infinite ease-in-out; animation-delay: 0s; margin-right: 0;"></div>
        </div>
    `;
    chatMessages.appendChild(typingDiv);
    chatMessages.scrollTop = chatMessages.scrollHeight;
    isTyping = true;
}

function hideTypingIndicator() {
    const typingIndicator = document.getElementById('typing-indicator-container');
    if (typingIndicator) {
        typingIndicator.remove();
    }
    isTyping = false;
}

function setLoadingState(loading) {
    const sendBtn = document.getElementById('send-btn');
    const input = document.getElementById('chat-input');
    
    if (loading) {
        sendBtn.disabled = true;
        sendBtn.innerHTML = `
            <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24" style="animation: spin 1s linear infinite;">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
            </svg>
            <span>Thinking...</span>
        `;
        input.disabled = true;
    } else {
        sendBtn.disabled = false;
        sendBtn.innerHTML = `
            <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"></path>
            </svg>
            <span>Send</span>
        `;
        input.disabled = false;
    }
}

function handleKeyPress(event) {
    if (event.key === 'Enter' && !event.shiftKey) {
        event.preventDefault();
        sendMessage();
    }
}

function showSessionMessage(message) {
    const container = document.getElementById('session-message-container');
    const textElement = document.getElementById('session-message-text');
    
    if (container && textElement) {
        textElement.textContent = message;
        container.style.display = 'flex';
    }
}

function hideSessionMessage() {
    const container = document.getElementById('session-message-container');
    if (container) {
        container.style.display = 'none';
    }
}

function insertQuickMessage(message) {
    const input = document.getElementById('chat-input');
    input.value = message;
    input.focus();
    input.style.height = 'auto';
    input.style.height = input.scrollHeight + 'px';
}

function clearCurrentChat() {
    if (confirm('Are you sure you want to clear this chat?')) {
        // Implement clear chat functionality
        const chatMessages = document.getElementById('chat-messages');
        chatMessages.innerHTML = `
            <div class="pc-message pc-message-ai">
                <div class="pc-avatar">
                    <svg width="20" height="20" fill="currentColor" viewBox="0 0 20 20">
                        <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                </div>
                <div class="pc-message-content">
                    <div class="pc-message-text">
                        <p><strong>🔄 Chat cleared!</strong></p>
                        <p>How can I assist you today?</p>
                    </div>
                    <span class="pc-timestamp">${getCurrentTime()}</span>
                </div>
            </div>
        `;
        messageCount = 0;
    }
}

function exportChat() {
    const messages = document.querySelectorAll('.pc-message-text');
    let chatContent = 'Anna Chat Export\n';
    chatContent += '='.repeat(30) + '\n\n';
    
    messages.forEach((msg, index) => {
        const isUser = msg.closest('.pc-message-user') !== null;
        const sender = isUser ? 'You' : 'Anna';
        chatContent += `${sender}: ${msg.textContent}\n\n`;
    });
    
    const blob = new Blob([chatContent], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `anna-chat-${new Date().toISOString().split('T')[0]}.txt`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
}

function renameConversation(conversationId) {
    const currentItem = document.querySelector(`[data-conversation-id="${conversationId}"]`);
    const titleElement = currentItem.querySelector('.pc-conversation-title');
    const currentTitle = titleElement.dataset.originalTitle;
    
    const newTitle = prompt('Enter new conversation title:', currentTitle);
    if (newTitle && newTitle !== currentTitle) {
        const formData = new FormData();
        formData.append('title', newTitle);
        formData.append('<?php echo $this->security->get_csrf_token_name(); ?>', getCsrfToken());
        
        fetch(`<?php echo admin_url('anna/rename_conversation/'); ?>${conversationId}`, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                titleElement.textContent = newTitle;
                titleElement.dataset.originalTitle = newTitle;
                if (conversationId == currentConversationId) {
                    document.getElementById('current-chat-title').textContent = newTitle;
                }
            } else {
                alert('Failed to rename conversation');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error renaming conversation');
        });
    }
}

function deleteConversation(conversationId) {
    if (confirm('Are you sure you want to delete this conversation? This action cannot be undone.')) {
        // Show loading state
        const deleteBtn = document.querySelector(`[data-conversation-id="${conversationId}"] .pc-action-btn[title="Delete"]`);
        if (deleteBtn) {
            deleteBtn.innerHTML = '<svg width="12" height="12" fill="currentColor" viewBox="0 0 20 20"><path d="M4 2a1 1 0 011 1v1h10V3a1 1 0 112 0v1h3a1 1 0 110 2h-1v12a2 2 0 01-2 2H7a2 2 0 01-2-2V6H4a1 1 0 110-2h3V3a1 1 0 011-1z"></path></svg>';
            deleteBtn.disabled = true;
        }
        
        fetch(`<?php echo admin_url('anna/delete_conversation/'); ?>${conversationId}`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: '<?php echo $this->security->get_csrf_token_name(); ?>=' + encodeURIComponent(getCsrfToken())
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                // Remove the conversation item from the list
                const item = document.querySelector(`[data-conversation-id="${conversationId}"]`);
                if (item) {
                    item.style.transition = 'opacity 0.3s ease';
                    item.style.opacity = '0';
                    setTimeout(() => {
                        item.remove();
                        // Check if this was the last conversation
                        const remainingConversations = document.querySelectorAll('.pc-conversation-item');
                        if (remainingConversations.length === 0) {
                            document.querySelector('.pc-conversations-list').innerHTML = 
                                '<div class="pc-no-conversations"><p>No conversations yet. Start a new chat!</p></div>';
                        }
                    }, 300);
                }
                
                // If we're deleting the current conversation, redirect to new chat
                if (conversationId == currentConversationId) {
                    window.location.href = '<?php echo admin_url('anna/chat'); ?>';
                }
                
                // Show success message (use Perfex's alert system if available)
                if (typeof alert_float === 'function') {
                    alert_float('success', 'Conversation deleted successfully');
                }
            } else {
                console.error('Delete failed:', data);
                alert('Failed to delete conversation: ' + (data.error || 'Unknown error'));
                // Restore button
                if (deleteBtn) {
                    deleteBtn.innerHTML = '<svg width="12" height="12" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path></svg>';
                    deleteBtn.disabled = false;
                }
            }
        })
        .catch(error => {
            console.error('Delete error:', error);
            alert('Error deleting conversation: ' + error.message);
            // Restore button
            if (deleteBtn) {
                deleteBtn.innerHTML = '<svg width="12" height="12" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path></svg>';
                deleteBtn.disabled = false;
            }
        });
    }
}

function updateSidebarAfterMessage() {
    // Update message count in sidebar
    const currentItem = document.querySelector(`[data-conversation-id="${currentConversationId}"]`);
    if (currentItem) {
        const messageCountSpan = currentItem.querySelector('.pc-message-count');
        if (messageCountSpan) {
            const current = parseInt(messageCountSpan.textContent);
            messageCountSpan.textContent = (current + 2) + ' messages'; // +2 for user and AI response
        }
    }
}

function getCurrentTime() {
    return new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
}

function getCsrfToken() {
    const token = document.querySelector('input[name="<?php echo $this->security->get_csrf_token_name(); ?>"]');
    return token ? token.value : '';
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatMessage(message) {
    message = escapeHtml(message);
    
    // Bold text (**text**)
    message = message.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
    
    // Bullet points (• text)
    message = message.replace(/^• (.+)$/gm, '<div class="pc-capability"><span class="pc-bullet">•</span><span>$1</span></div>');
    
    // Line breaks
    message = message.replace(/\n/g, '<br>');
    
    // Code blocks (`code`)
    message = message.replace(/`([^`]+)`/g, '<code style="background: #f3f4f6; padding: 2px 4px; border-radius: 3px; font-size: 0.9em;">$1</code>');
    
    return message;
}

// Confirmation handling functions
function displayConfirmationRequest(data) {
    const chatMessages = document.getElementById('chat-messages');
    const messageDiv = document.createElement('div');
    const currentTime = getCurrentTime();
    
    messageDiv.className = 'pc-message pc-message-ai';
    
    let summaryHtml = '';
    if (data.action_type === 'create_invoice') {
        summaryHtml = createInvoiceSummaryHtml(data.summary);
    } else if (data.action_type === 'create_customer') {
        summaryHtml = createCustomerSummaryHtml(data.summary);
    }
    
    messageDiv.innerHTML = `
        <div class="pc-avatar">
            <svg width="20" height="20" fill="currentColor" viewBox="0 0 20 20">
                <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
        </div>
        <div class="pc-message-content">
            <div class="pc-confirmation-request" style="background: #fef3c7; border: 1px solid #f59e0b; border-radius: 8px; padding: 16px; margin: 8px 0;">
                <div style="display: flex; align-items: center; margin-bottom: 12px;">
                    <svg width="20" height="20" fill="#f59e0b" viewBox="0 0 20 20" style="margin-right: 8px;">
                        <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"></path>
                    </svg>
                    <h5 style="margin: 0; color: #92400e; font-weight: 600;">${data.confirmation_ui.title}</h5>
                </div>
                <p style="margin: 8px 0; color: #92400e;">${data.message}</p>
                ${summaryHtml}
                <div style="display: flex; gap: 8px; margin-top: 16px;">
                    <button onclick="handleConfirmation('${data.confirmation_id}', 'accept')" 
                            class="btn btn-success btn-sm" style="background: #059669; border: none;">
                        <i class="fa fa-check"></i> ${data.confirmation_ui.accept_button}
                    </button>
                    <button onclick="handleConfirmation('${data.confirmation_id}', 'reject')" 
                            class="btn btn-danger btn-sm" style="background: #dc2626; border: none;">
                        <i class="fa fa-times"></i> ${data.confirmation_ui.reject_button}
                    </button>
                </div>
            </div>
            <span class="pc-timestamp">${currentTime}</span>
        </div>
    `;
    
    chatMessages.appendChild(messageDiv);
    chatMessages.scrollTop = chatMessages.scrollHeight;
}

function createInvoiceSummaryHtml(summary) {
    let itemsHtml = '';
    summary.items.forEach(item => {
        itemsHtml += `
            <tr>
                <td>${item.description}</td>
                <td>${item.quantity}</td>
                <td>$${item.rate}</td>
                <td>${item.tax_rate}</td>
                <td>$${item.total}</td>
            </tr>
        `;
    });
    
    return `
        <div style="background: white; padding: 12px; border-radius: 6px; margin: 12px 0;">
            <h6 style="margin: 0 0 8px 0; color: #374151;">Invoice Summary</h6>
            <div style="margin-bottom: 8px;">
                <strong>Customer:</strong> ${summary.customer_info.company} (ID: ${summary.customer_info.id})
            </div>
            <div style="margin-bottom: 8px;">
                <strong>Due Date:</strong> ${summary.invoice_details.due_date}
            </div>
            <table style="width: 100%; border-collapse: collapse; font-size: 12px; margin: 8px 0;">
                <thead>
                    <tr style="background: #f9fafb;">
                        <th style="padding: 4px; text-align: left; border: 1px solid #e5e7eb;">Description</th>
                        <th style="padding: 4px; text-align: center; border: 1px solid #e5e7eb;">Qty</th>
                        <th style="padding: 4px; text-align: right; border: 1px solid #e5e7eb;">Rate</th>
                        <th style="padding: 4px; text-align: center; border: 1px solid #e5e7eb;">Tax</th>
                        <th style="padding: 4px; text-align: right; border: 1px solid #e5e7eb;">Total</th>
                    </tr>
                </thead>
                <tbody>
                    ${itemsHtml}
                </tbody>
            </table>
            <div style="text-align: right; font-weight: bold; padding-top: 8px; border-top: 1px solid #e5e7eb;">
                Total: $${summary.invoice_details.total_amount}
            </div>
        </div>
    `;
}

function createCustomerSummaryHtml(summary) {
    return `
        <div style="background: white; padding: 12px; border-radius: 6px; margin: 12px 0;">
            <h6 style="margin: 0 0 8px 0; color: #374151;">Customer Summary</h6>
            <div style="margin-bottom: 4px;"><strong>Company:</strong> ${summary.customer_info.company}</div>
            <div style="margin-bottom: 4px;"><strong>Website:</strong> ${summary.customer_info.website || 'Not provided'}</div>
            <div style="margin-bottom: 4px;"><strong>Phone:</strong> ${summary.customer_info.phone || 'Not provided'}</div>
            <div style="margin-bottom: 4px;"><strong>VAT Number:</strong> ${summary.customer_info.vat_number || 'Not provided'}</div>
            ${summary.primary_contact ? `
                <div style="margin-top: 8px; padding-top: 8px; border-top: 1px solid #e5e7eb;">
                    <strong>Primary Contact:</strong> ${summary.primary_contact.firstname} ${summary.primary_contact.lastname}<br>
                    <strong>Email:</strong> ${summary.primary_contact.email}
                </div>
            ` : ''}
        </div>
    `;
}

function handleConfirmation(confirmationId, action) {
    const confirmationDiv = document.querySelector(`[onclick*="${confirmationId}"]`).closest('.pc-confirmation-request');
    const buttons = confirmationDiv.querySelectorAll('button');
    
    // Disable buttons and show loading
    buttons.forEach(btn => {
        btn.disabled = true;
        if (btn.onclick.toString().includes(action)) {
            btn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Processing...';
        }
    });
    
    const formData = new FormData();
    formData.append('confirmation_id', confirmationId);
    formData.append('action', action);
    formData.append('<?php echo $this->security->get_csrf_token_name(); ?>', getCsrfToken());
    
    fetch('<?php echo admin_url('anna/handle_confirmation'); ?>', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Replace confirmation with success message
            confirmationDiv.innerHTML = `
                <div style="background: #d1fae5; border: 1px solid #059669; border-radius: 6px; padding: 12px; color: #065f46;">
                    <div style="display: flex; align-items: center;">
                        <svg width="20" height="20" fill="#059669" viewBox="0 0 20 20" style="margin-right: 8px;">
                            <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"></path>
                        </svg>
                        <strong>${action === 'accept' ? 'Action Completed' : 'Action Cancelled'}</strong>
                    </div>
                    <p style="margin: 8px 0 0 28px;">${data.message}</p>
                    ${data.invoice_url ? `<a href="${data.invoice_url}" target="_blank" style="margin-left: 28px; color: #059669;">View Invoice</a>` : ''}
                    ${data.customer_url ? `<a href="${data.customer_url}" target="_blank" style="margin-left: 28px; color: #059669;">View Customer</a>` : ''}
                </div>
            `;
        } else {
            // Show error and re-enable buttons
            alert('Error: ' + (data.error || 'Unknown error occurred'));
            buttons.forEach(btn => {
                btn.disabled = false;
                if (btn.onclick.toString().includes('accept')) {
                    btn.innerHTML = '<i class="fa fa-check"></i> Create Invoice';
                } else {
                    btn.innerHTML = '<i class="fa fa-times"></i> Cancel';
                }
            });
        }
    })
    .catch(error => {
        console.error('Confirmation error:', error);
        alert('Error processing confirmation. Please try again.');
        // Re-enable buttons
        buttons.forEach(btn => {
            btn.disabled = false;
            if (btn.onclick.toString().includes('accept')) {
                btn.innerHTML = '<i class="fa fa-check"></i> Create Invoice';
            } else {
                btn.innerHTML = '<i class="fa fa-times"></i> Cancel';
            }
        });
    });
}

// Add CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }
    
    @keyframes typing {
        0%, 80%, 100% {
            opacity: 0.3;
            transform: scale(0.8);
        }
        40% {
            opacity: 1;
            transform: scale(1);
        }
    }
`;
document.head.appendChild(style);
</script>

<?php init_tail(); ?>