<?php

namespace TourGuide\Http\Controllers;

/**
 * TourGuideAdminControllerTrait
 *
 * Handles CRUD operations for the tour guide feature within the admin panel.
 * Provides functionality for viewing, creating, updating, and deleting tour guides.
 */
trait TourGuideAdminControllerTrait
{
    /**
     * Display the list of tour guides.
     *
     * Ensures the user has permission to view tour guides before loading the view.
     */
    public function index()
    {
        $this->ensureStaffCan("view");

        $data['title'] = tourGuideHelper()->translate('tour_guide');
        $data['tour_guides'] =  [];

        try {
            $data['tour_guides'] = array_values((array)$this->tourGuideReposiotry->getAllTours());
        } catch (\Throwable $th) {
            $handler = $this->response([
                'success' => false,
                'message' => $th->getMessage(),
            ]);
            $this->setAlert($handler['status'], $handler['message']);
        }

        return $this->loadView('admin/manage', $data);
    }

    /**
     * Display the form for creating or editing a tour guide.
     *
     * @param int $id Optional ID of the tour guide to edit. If not provided, the form is for creating a new guide.
     */
    public function form($id = '')
    {
        $id = (int)$id;

        // Check if the user has permission to create or edit based on the presence of an ID
        $this->ensureStaffCan($id ? 'edit' : 'create', $id);

        $handler = $this->handleForm($id);
        if ($handler['is_post']) {
            $this->setAlert($handler['status'], $handler['message']);

            $url =  tourGuideHelper()->adminUrl();
            if (!$handler['success']) {
                $url = tourGuideHelper()->adminUrl('form/' . $id);
            }
            return $this->redirect($url);
        }

        $data = $handler['data'];

        return $this->loadView('admin/form', $data);
    }

    /**
     * Delete a tour guide by ID.
     *
     * @param int $id ID of the tour guide to delete.
     */
    public function delete($id)
    {
        $this->ensureStaffCan('delete', $id);

        $handler = $this->handleDelete($id);

        if ($this->isAJAX()) {

            return $this->responseJson($handler);
        }

        $this->setAlert($handler['status'], $handler['message']);
        return $this->redirect(tourGuideHelper()->adminUrl());
    }

    /**
     * Clone a tour guide by ID.
     *
     * @param int $id ID of the tour guide to clone.
     */
    public function clone($id)
    {
        $this->ensureStaffCan('create', $id);

        $handler = $this->handleClone($id);

        if ($this->isAJAX()) {

            return $this->responseJson($handler);
        }

        $this->setAlert($handler['status'], $handler['message']);
        return $this->redirect(tourGuideHelper()->adminUrl());
    }

    /**
     * Translate a tour identified by the $id
     *
     * @param string $id
     * @return void
     */
    public function translate($id)
    {
        // Require edit permission
        $this->ensureStaffCan('edit', $id);

        $handler = $this->handleTranslate($id);
        if ($this->isAJAX()) {

            return $this->responseJson($handler);
        }

        if (!empty($handler['message']))
            $this->setAlert($handler['status'], $handler['message']);

        if (!empty($handler['redirect'])) {
            return $this->redirect($handler['redirect']);
        }

        $data = $handler['data'];
        $data['handler'] = $handler;

        return $this->loadView('admin/translate', $data);
    }
}