<?php

namespace TourGuide\Http\Controllers;

use TourGuide\Widgets\TourGuideWidgetRepository;
use TourGuide\Widgets\TourGuideWidgetUtils;
use TourGuide\TourGuideUtils;
use TourGuide\TourGuideRepository;

/**
 * TourGuideWidgetAdminControllerTrait
 *
 * Handles widget CRUD operations for the admin panel.
 * Provides functionality for viewing, creating, updating, and deleting widgets.
 */
trait TourGuideWidgetAdminControllerTrait
{
    /**
     * Display the list of widgets.
     */
    public function widgets($action = '', $id = '')
    {
        $this->ensureStaffCan("view");

        if (empty($action)) {
            return $this->widgetsList();
        }

        switch ($action) {
            case 'create':
                return $this->widgetsCreate();
            case 'edit':
                return $this->widgetsEdit($id);
            case 'delete':
                return $this->widgetsDelete($id);
            default:
                return $this->widgetsList();
        }
    }

    /**
     * Display widgets list.
     */
    private function widgetsList()
    {
        $data['title'] = tourGuideHelper()->translate('tour_guide_widgets');
        $data['widgets'] = [];

        try {
            $widgetRepository = new TourGuideWidgetRepository();
            $data['widgets'] = array_values((array)$widgetRepository->getAllWidgets());
        } catch (\Throwable $th) {
            $this->setAlert('danger', $th->getMessage());
        }

        return $this->loadView('admin/widgets_manage', $data);
    }

    /**
     * Create new widget.
     */
    private function widgetsCreate()
    {
        $this->ensureStaffCan("create");

        $handler = $this->handleWidgetForm();

        if ($this->isAJAX()) {
            return $this->responseJson($handler);
        }

        if (!empty($handler['message'])) {
            $this->setAlert($handler['status'], $handler['message']);
        }

        if (!empty($handler['redirect'])) {
            return $this->redirect($handler['redirect']);
        }

        $data = $handler['data'];
        $data['handler'] = $handler;

        return $this->loadView('admin/widgets_form', $data);
    }

    /**
     * Edit existing widget.
     */
    private function widgetsEdit($id)
    {
        $this->ensureStaffCan("edit");

        $handler = $this->handleWidgetForm($id);

        if ($this->isAJAX()) {
            return $this->responseJson($handler);
        }

        if (!empty($handler['message'])) {
            $this->setAlert($handler['status'], $handler['message']);
        }

        if (!empty($handler['redirect'])) {
            return $this->redirect($handler['redirect']);
        }

        $data = $handler['data'];
        $data['handler'] = $handler;

        return $this->loadView('admin/widgets_form', $data);
    }

    /**
     * Delete widget.
     */
    private function widgetsDelete($id)
    {
        $this->ensureStaffCan("delete");

        $handler = [
            'success' => false,
            'message' => '',
            'status' => 'danger',
            'redirect' => '',
            'data' => []
        ];

        try {
            $widgetRepository = new TourGuideWidgetRepository();

            if ($widgetRepository->delete($id)) {
                $handler['success'] = true;
                $handler['status'] = 'success';
                $handler['message'] = tourGuideHelper()->translate('tour_guide_widget_deleted_successfully');
                $handler['redirect'] = $this->getWidgetsUrl();
            } else {
                $handler['message'] = tourGuideHelper()->translate('tour_guide_widget_delete_failed');
            }
        } catch (\Throwable $th) {
            $handler['message'] = $th->getMessage();
        }

        if ($this->isAJAX()) {
            return $this->responseJson($handler);
        }

        $this->setAlert($handler['status'], $handler['message']);
        return $this->redirect($handler['redirect'] ?: $this->getWidgetsUrl());
    }

    /**
     * Handle widget form submission and display.
     */
    private function handleWidgetForm($id = null)
    {
        $handler = [
            'success' => false,
            'message' => '',
            'status' => 'danger',
            'redirect' => '',
            'data' => [
                'title' => $id ? tourGuideHelper()->translate('tour_guide_edit_widget') : tourGuideHelper()->translate('tour_guide_create_widget'),
                'widget' => null,
                'tours_list' => []
            ]
        ];

        try {
            $widgetRepository = new TourGuideWidgetRepository();

            // Get tours list for selection
            $tourRepository = new TourGuideRepository();
            $handler['data']['tours_list'] = array_values((array)$tourRepository->getAllTours());

            // If editing, get the widget
            if ($id) {
                $widget = $widgetRepository->getWidget($id);
                if (!$widget) {
                    throw new \Exception(tourGuideHelper()->translate('tour_guide_widget_not_found'));
                }
                $handler['data']['widget'] = $widget;
            }

            // Handle form submission
            if (TourGuideUtils::isPostRequest()) {
                $formData = TourGuideWidgetUtils::getWidgetFormData();

                if ($id) {
                    $formData['id'] = $id;
                }

                $widgetId = $widgetRepository->save($formData);

                $handler['success'] = true;
                $handler['status'] = 'success';
                $handler['message'] = $id ?
                    tourGuideHelper()->translate('tour_guide_widget_updated_successfully') :
                    tourGuideHelper()->translate('tour_guide_widget_created_successfully');
                $handler['redirect'] = $this->getWidgetsUrl();
            }
        } catch (\Throwable $th) {
            $handler['message'] = $th->getMessage();
        }

        return $handler;
    }

    /**
     * Get widgets management URL.
     */
    private function getWidgetsUrl()
    {
        return tourGuideHelper()->adminUrl('widgets');
    }
}