<?php

namespace TourGuide\Widgets;

use TourGuide\Widgets\Widget;
use TourGuide\TourGuideRepository;

/**
 * TourGuideWidgetAPI
 *
 * Handles public API endpoints for widget system.
 * Serves widget JavaScript files and handles analytics.
 */
class TourGuideWidgetAPI
{
    /** @var TourGuideWidgetRepository $repository */
    private $repository;

    /** @var TourGuideRepository $tourRepository */
    private $tourRepository;

    function __construct()
    {
        $this->repository = new TourGuideWidgetRepository();
        $this->tourRepository = new TourGuideRepository();
    }

    /**
     * Send HTTP response with proper headers and content.
     *
     * @param mixed $content Response content
     * @param int $statusCode HTTP status code
     * @param string $contentType Content-Type header
     * @param array $headers Additional headers
     * @return void
     */
    private function sendResponse($content, $statusCode = 200, $contentType = 'text/plain', $headers = [])
    {
        http_response_code($statusCode);
        header("Content-Type: {$contentType}");

        foreach ($headers as $header => $value) {
            header("{$header}: {$value}");
        }

        echo $content;
    }

    /**
     * Send JSON response.
     *
     * @param mixed $data Data to encode as JSON
     * @param int $statusCode HTTP status code
     * @param array $headers Additional headers
     * @return void
     */
    private function sendJsonResponse($data, $statusCode = 200, $headers = [])
    {
        $this->sendResponse(json_encode($data), $statusCode, 'application/json', $headers);
    }

    /**
     * Send JavaScript response.
     *
     * @param string $js JavaScript content
     * @param int $statusCode HTTP status code
     * @param bool $cacheable Whether to set cache headers
     * @return void
     */
    private function sendJavaScriptResponse($js, $statusCode = 200, $cacheable = true)
    {
        $headers = [];
        if ($cacheable) {
            $headers['Cache-Control'] = 'public, max-age=3600';
            $headers['Expires'] = gmdate('D, d M Y H:i:s', time() + 3600) . ' GMT';
        }

        $this->sendResponse($js, $statusCode, 'application/javascript', $headers);
    }


    /**
     * Send tracking pixel response.
     *
     * @return void
     */
    private function sendTrackingPixelResponse()
    {
        $headers = [
            'Cache-Control' => 'no-cache, no-store, must-revalidate',
            'Pragma' => 'no-cache',
            'Expires' => '0'
        ];

        // 1x1 transparent GIF
        $pixel = base64_decode('R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7');
        $this->sendResponse($pixel, 200, 'image/gif', $headers);
    }

    /**
     * Send error response.
     *
     * @param string $message Error message
     * @param int $statusCode HTTP status code
     * @param string $contentType Content-Type header
     * @return void
     */
    private function sendErrorResponse($message, $statusCode = 500, $contentType = 'text/plain')
    {
        $this->sendResponse($message, $statusCode, $contentType);
    }

    /**
     * Serve widget JavaScript file.
     *
     * @param string $widget_id Widget identifier.
     * @param string $baseUrl Base URL for assets.
     * @return void
     */
    public function serveWidgetJS($widget_id, $baseUrl = '')
    {
        try {
            // Get widget configuration
            $widget = $this->repository->getWidgetByWidgetId($widget_id);

            if (!$widget || $widget->status !== 'active') {
                $this->sendJavaScriptResponse('// Widget not found or inactive', 404, false);
                return;
            }

            // Validate domain if whitelist is set
            $origin  = $_SERVER['HTTP_ORIGIN']  ?? '';
            $referer = $_SERVER['HTTP_REFERER'] ?? '';

            $source = $origin ?: $referer;
            $domain = parse_url($source, PHP_URL_HOST) ?: '';

            // Validate against whitelist
            if (!empty($widget->domain_whitelist) && !TourGuideWidgetUtils::validateDomain($domain, $widget->domain_whitelist)) {
                $this->sendJavaScriptResponse('// Domain not allowed: ' . $domain, 403, false);
                return;
            }


            // Get tours data for the widget
            $tours = [];
            if (!empty($widget->selected_tours)) {
                foreach ($widget->selected_tours as $tour_id) {
                    $tour = $this->tourRepository->getTour($tour_id);
                    if ($tour && $tour->status === 'active') {
                        $tours[] = $tour;
                    }
                }
            }

            // Prepare widget configuration
            $widgetConfig = [
                'widget_id' => $widget->widget_id,
                'type' => $widget->type,
                'title' => $widget->title,
                'tours' => $tours,
                'settings' => array_merge($widget->settings, [
                    'baseUrl' => $baseUrl,
                    'apiUrl' => rtrim($baseUrl, '/') . '/api/widget'
                ])
            ];

            // Generate and serve JavaScript
            $js = TourGuideWidgetUtils::generateWidgetJS($widgetConfig, $baseUrl);
            $minifiedJS = TourGuideWidgetUtils::minifyJS($js);

            $this->sendJavaScriptResponse($minifiedJS);
        } catch (\Exception $e) {
            $this->sendJavaScriptResponse('// Error loading widget: ' . addslashes($e->getMessage()), 500, false);
        }
    }

    /**
     * Handle widget analytics tracking.
     *
     * @return void
     */
    public function handleAnalytics()
    {
        try {
            // Get analytics data from query parameters or POST
            $data = array_merge($_GET, $_POST);

            $widget_id = $data['widget_id'] ?? '';
            $event_type = $data['event_type'] ?? '';

            if (empty($widget_id) || empty($event_type)) {
                $this->sendErrorResponse('Missing required parameters', 400);
                return;
            }

            // Validate widget exists
            $widget = $this->repository->getWidgetByWidgetId($widget_id);
            if (!$widget) {
                $this->sendErrorResponse('Widget not found', 404);
                return;
            }

            // Log the analytics event
            $this->repository->logAnalytics($widget_id, $event_type, [
                'domain' => $data['domain'] ?? '',
                'page_url' => $data['page_url'] ?? '',
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'tour_id' => $data['tour_id'] ?? null
            ]);

            // Return 1x1 transparent pixel for image-based tracking
            $this->sendTrackingPixelResponse();
        } catch (\Exception $e) {
            $this->sendErrorResponse('Analytics error: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get widget configuration for AJAX requests.
     *
     * @param string $widget_id Widget identifier.
     * @return void
     */
    public function getWidgetConfig($widget_id)
    {
        try {
            $widget = $this->repository->getWidgetByWidgetId($widget_id);

            if (!$widget || $widget->status !== 'active') {
                $this->sendJsonResponse(['error' => 'Widget not found'], 404);
                return;
            }

            // Validate domain
            $domain = $_SERVER['HTTP_HOST'] ?? '';
            if (!TourGuideWidgetUtils::validateDomain($domain, $widget->domain_whitelist)) {
                $this->sendJsonResponse(['error' => 'Domain not allowed'], 403);
                return;
            }

            // Get tours for the widget
            $tours = [];
            if (!empty($widget->selected_tours)) {
                foreach ($widget->selected_tours as $tour_id) {
                    $tour = $this->tourRepository->getTour($tour_id);
                    if ($tour && $tour->status === 'active') {
                        $tours[] = $tour;
                    }
                }
            }

            $this->sendJsonResponse([
                'widget_id' => $widget->widget_id,
                'type' => $widget->type,
                'title' => $widget->title,
                'tours' => $tours,
                'settings' => $widget->settings
            ]);
        } catch (\Exception $e) {
            $this->sendJsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Handle widget API routing.
     *
     * @param string $endpoint The API endpoint.
     * @param array $params URL parameters.
     * @param string $baseUrl Base URL for assets.
     * @return void
     */
    public static function handleRequest($endpoint, $params = [], $baseUrl = '')
    {
        $api = new self();

        $baseUrl = $baseUrl ?: $baseUrl = tourGuideHelper()->userUrl('widget');

        switch ($endpoint) {
            case 'widget':
                // Serve widget JS: /api/widget/{widget_id}.js
                $widget_id = $params['widget_id'] ?? '';
                if (empty($widget_id)) {
                    $api->sendErrorResponse('Widget ID required', 400);
                    return;
                }
                $api->serveWidgetJS($widget_id, $baseUrl);
                break;

            case 'analytics':
                // Handle analytics: /api/widget/analytics
                $api->handleAnalytics();
                break;

            case 'config':
                // Get widget config: /api/widget/config/{widget_id}
                $widget_id = $params['widget_id'] ?? '';
                if (empty($widget_id)) {
                    $api->sendErrorResponse('Widget ID required', 400);
                    return;
                }
                $api->getWidgetConfig($widget_id);
                break;

            default:
                $api->sendErrorResponse('API endpoint not found', 404);
                break;
        }
    }
}