<?php

namespace TourGuide\Widgets;

use stdClass;
use TourGuide\Widgets\Widget;

/**
 * TourGuideWidgetRepository
 *
 * Handles database interactions for the widget system.
 * Provides methods for retrieving, adding, updating, and deleting widget records.
 */
class TourGuideWidgetRepository
{
    /** @var TourGuideDB $db */
    private $db;

    private $table;
    private $analyticsTable;

    function __construct()
    {
        $this->db = tourGuideHelper()->dbInstance();
        $this->table = 'tour_guide_widgets';
        $this->analyticsTable = 'tour_guide_widget_analytics';
    }

    /**
     * Get all widgets mapped and cleaned
     *
     * @return object
     */
    public function getAllWidgets()
    {
        $widgets_id_map = new stdClass;
        $widgets = $this->getAll();

        foreach ($widgets as $widget_data) {
            $widget = new Widget($widget_data);
            $widgets_id_map->{$widget->widget_id} = $widget->toArray();
        }

        return $widgets_id_map;
    }

    /**
     * Retrieve a specific widget by its widget_id.
     *
     * @param string $widget_id The widget_id of the widget to retrieve.
     * @return object|null The widget as an object.
     */
    public function getWidgetByWidgetId($widget_id)
    {
        $widget = $this->getByWidgetId($widget_id, "array");

        if (!$widget) return null;

        $widgetModel = new Widget($widget);
        return (object)$widgetModel->toArray();
    }

    /**
     * Retrieve a specific widget by its database ID.
     *
     * @param int $id The ID of the widget to retrieve.
     * @return object|null The widget as an object.
     */
    public function getWidget($id)
    {
        $widget = $this->get($id, "array");

        if (!$widget) return null;

        $widgetModel = new Widget($widget);
        return (object)$widgetModel->toArray();
    }

    /**
     * Retrieve all widget records from the database.
     *
     * @return array An array of widget records.
     */
    public function getAll()
    {
        return $this->db->read($this->table, [], '*', 'created_at DESC');
    }

    /**
     * Retrieve active widgets only.
     *
     * @return array An array of active widget records.
     */
    public function getActiveWidgets()
    {
        return $this->db->read($this->table, ['status' => 'active'], '*', 'created_at DESC');
    }

    /**
     * Retrieve widgets by type.
     *
     * @param string $type Widget type ('setup' or 'player').
     * @return array An array of widget records.
     */
    public function getWidgetsByType($type)
    {
        return $this->db->read($this->table, ['type' => $type, 'status' => 'active'], '*', 'created_at DESC');
    }

    /**
     * Retrieve a specific widget by its database ID.
     *
     * @param int $id The ID of the widget to retrieve.
     * @param string $format Return format ('object' or 'array').
     * @return object|array|null The widget record.
     */
    public function get($id, $format = 'object')
    {
        $results = $this->db->read($this->table, ['id' => $id]);
        if (empty($results)) return null;
        return $format == 'object' ? (object)$results[0] : $results[0];
    }

    /**
     * Retrieve a specific widget by its widget_id.
     *
     * @param string $widget_id The widget_id to search for.
     * @param string $format Return format ('object' or 'array').
     * @return object|array|null The widget record.
     */
    public function getByWidgetId($widget_id, $format = 'object')
    {
        $results = $this->db->read($this->table, ['widget_id' => $widget_id]);
        if (empty($results)) return null;
        return $format == 'object' ? (object)$results[0] : $results[0];
    }

    /**
     * Save a widget record to the database.
     *
     * @param array $data The widget data to save.
     * @return int The ID of the inserted or updated record.
     */
    public function save($data)
    {
        // Validate data
        $errors = Widget::validate($data);
        if (!empty($errors)) {
            throw new \Exception(implode(', ', $errors));
        }

        // Prepare data for saving
        $data = Widget::prepareForSaving($data);

        if (!empty($data['id'])) {
            unset($data['widget_id']);
            return $this->update($data, $data['id']);
        } else {
            return $this->add($data);
        }
    }

    /**
     * Add a new widget record to the database.
     *
     * @param array $data The widget data to add.
     * @return int The ID of the newly inserted record.
     */
    public function add($data)
    {
        return $this->db->create($this->table, $data);
    }

    /**
     * Update an existing widget record in the database.
     *
     * @param array $data The widget data to update.
     * @param int $id The ID of the widget to update.
     * @return int The number of affected rows.
     */
    public function update($data, $id)
    {
        // Remove id from data to avoid updating it
        unset($data['id']);
        return $this->db->update($this->table, $data, ['id' => $id]);
    }

    /**
     * Clone an existing widget record.
     *
     * @param int $id The ID of the widget to clone.
     * @return int The ID of the new widget.
     */
    public function clone($id)
    {
        $widget = $this->get($id, 'array');
        if (!$widget) {
            throw new \Exception('Widget not found');
        }

        $data = $widget;
        $data['title'] .= ' #copy';
        $data['widget_id'] = Widget::generateWidgetId();

        unset($data['id']);
        unset($data['created_at']);
        unset($data['updated_at']);

        return $this->save($data);
    }

    /**
     * Delete a widget record from the database.
     *
     * @param int $id The ID of the widget to delete.
     * @return int The number of affected rows.
     */
    public function delete($id)
    {
        // Get widget data before deletion for cleanup
        $widget = $this->get($id, 'array');

        if ($widget) {
            // Delete associated analytics first
            $this->db->delete($this->analyticsTable, ['widget_id' => $widget['widget_id']]);
        }

        return $this->db->delete($this->table, ['id' => $id]);
    }

    /**
     * Log widget analytics event.
     *
     * @param string $widget_id Widget identifier.
     * @param string $event_type Type of event ('load', 'tour_start', 'tour_complete', 'tour_exit').
     * @param array $data Additional event data.
     * @return int The ID of the inserted analytics record.
     */
    public function logAnalytics($widget_id, $event_type, $data = [])
    {
        $analyticsData = [
            'widget_id' => $widget_id,
            'event_type' => $event_type,
            'domain' => $data['domain'] ?? $_SERVER['HTTP_HOST'] ?? '',
            'page_url' => $data['page_url'] ?? $_SERVER['REQUEST_URI'] ?? '',
            'user_agent' => $data['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip_address' => $this->getClientIpAddress(),
            'tour_id' => $data['tour_id'] ?? null,
        ];

        return $this->db->create($this->analyticsTable, $analyticsData);
    }

    /**
     * Get analytics data for a widget.
     *
     * @param string $widget_id Widget identifier.
     * @param int $limit Number of records to retrieve.
     * @return array Analytics data.
     */
    public function getAnalytics($widget_id, $limit = 100)
    {
        return $this->db->read(
            $this->analyticsTable,
            ['widget_id' => $widget_id],
            '*',
            'created_at DESC',
            $limit
        );
    }

    /**
     * Get widget usage statistics.
     *
     * @param string $widget_id Widget identifier.
     * @return array Usage statistics.
     */
    public function getUsageStats($widget_id)
    {
        $stats = [
            'total_loads' => 0,
            'total_starts' => 0,
            'total_completions' => 0,
            'completion_rate' => 0,
        ];

        $analytics = $this->getAnalytics($widget_id, 1000);

        foreach ($analytics as $event) {
            switch ($event['event_type']) {
                case 'load':
                    $stats['total_loads']++;
                    break;
                case 'tour_start':
                    $stats['total_starts']++;
                    break;
                case 'tour_complete':
                    $stats['total_completions']++;
                    break;
            }
        }

        if ($stats['total_starts'] > 0) {
            $stats['completion_rate'] = round(($stats['total_completions'] / $stats['total_starts']) * 100, 2);
        }

        return $stats;
    }

    /**
     * Get client IP address.
     *
     * @return string Client IP address.
     */
    private function getClientIpAddress()
    {
        $ipKeys = ['HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];

        foreach ($ipKeys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }

        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }
}