<?php

namespace TourGuide\Widgets;

use TourGuide\TourGuideUtils;
use TourGuide\TourGuideHook;

/**
 * Class TourGuideWidgetUIHelper
 *
 * A helper class for managing widget-specific UI components.
 * This class provides methods for rendering widget tables, forms, and related UI elements.
 *
 * @package TourGuide\Widgets
 *
 * @copyright Copyright (c) 2024, Ankabit Technologies. All rights reserved.
 * @license https://anakbit.com/legal/common-license
 *
 * @since 1.0
 */
class TourGuideWidgetUIHelper extends TourGuideUtils
{
    /**
     * Renders the widget table with customizable columns and row data.
     *
     * @param array $widgets Array of widget data.
     * @param string $orderCol Column index for default ordering.
     * @param string $orderType Type of ordering (asc/desc).
     * @param bool $includeBlankRow Whether to show empty table data message or not.
     */
    public static function renderWidgetTable(array $widgets = [], string $orderCol = '4', string $orderType = 'desc', bool $includeBlankRow = false): void
    {
        // Apply filter to modify table classes if needed.
        $tableClass = TourGuideHook::applyFilter('tour_guide_widget_table_class', 'table dt-table');

        echo '<div class="tour-guide-widgets-list">';
        echo '<table class="tour-guide-content-wrapper ' . self::escapeHtml($tableClass) . '"';
        echo ' data-order-type="' . $orderType . '" data-order-col="' . $orderCol . '">';
        echo '<thead>';
        echo '<tr>';

        // Define default columns
        $columns = [
            'title' => tourGuideHelper()->translate('tour_guide_widget_title'),
            'type' => tourGuideHelper()->translate('tour_guide_widget_type'),
            'status' => tourGuideHelper()->translate('tour_guide_status'),
            'widget_id' => tourGuideHelper()->translate('tour_guide_widget_id'),
            'created_at' => tourGuideHelper()->translate('tour_guide_created_at'),
            'actions' => tourGuideHelper()->translate('tour_guide_actions'),
        ];

        // Allow columns to be filtered/modified
        $columns = TourGuideHook::applyFilter('tour_guide_widget_table_columns', $columns);

        // Display each column
        foreach ($columns as $key => $label) {
            echo '<th>' . self::escapeHtml($label) . '</th>';
        }

        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        if (!empty($widgets)) {
            foreach ($widgets as $widget) {
                echo '<tr>';

                // Prepare the row data, allowing filtering/modification
                $rowData = [
                    'title' => self::renderWidgetTitle($widget),
                    'type' => self::renderWidgetType($widget),
                    'status' => self::renderWidgetStatus($widget),
                    'widget_id' => '<code class="tour-guide-widget-id">' . self::escapeHtml($widget['widget_id'] ?? '') . '</code>',
                    'created_at' => !empty($widget['created_at']) ? date('Y-m-d H:i', strtotime($widget['created_at'])) : '',
                    'actions' => self::renderWidgetTableRowActions($widget),
                ];

                // Allow row data to be modified by filters
                $rowData = TourGuideHook::applyFilter('tour_guide_widget_table_row', $rowData, $widget);

                // Display each column's data
                foreach ($columns as $key => $label) {
                    echo '<td>';
                    echo ($key == 'status' || $key == 'actions' || $key == 'widget_id' || $key == 'title' || $key == 'type') ?
                        $rowData[$key] : self::escapeHtml($rowData[$key] ?? '');
                    echo '</td>';
                }

                echo '</tr>';
            }
        } elseif ($includeBlankRow) {
            echo '<tr>';
            echo '<td colspan="' . count($columns) . '" class="text-center">';
            echo '<div class="tour-guide-empty-state">';
            echo '<p class="text-muted">' . tourGuideHelper()->translate('tour_guide_no_widgets_found') . '</p>';
            echo '</div>';
            echo '</td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
        echo '</div>';

        // Add embed code modal and JavaScript
        self::renderWidgetEmbedModal();
    }

    /**
     * Render widget title with description.
     */
    protected static function renderWidgetTitle(array $widget): string
    {
        $html = '<strong>' . self::escapeHtml($widget['title'] ?? '') . '</strong>';
        if (!empty($widget['description'])) {
            $html .= '<br><small class="text-muted">' . self::escapeHtml($widget['description']) . '</small>';
        }
        return $html;
    }

    /**
     * Render widget type badge.
     */
    protected static function renderWidgetType(array $widget): string
    {
        $type = $widget['type'] ?? '';
        $badgeClass = $type === 'setup' ? 'warning' : 'info';
        return '<span class="badge badge-' . $badgeClass . '">' .
            tourGuideHelper()->translate('tour_guide_widget_type_' . $type) . '</span>';
    }

    /**
     * Render widget status badge.
     */
    protected static function renderWidgetStatus(array $widget): string
    {
        $status = $widget['status'] ?? '';
        $badgeClass = $status === 'active' ? 'success' : 'danger';
        return '<span class="badge bg-' . $badgeClass . '">' .
            tourGuideHelper()->translate('tour_guide_' . $status) . '</span>';
    }

    /**
     * Render widget table row actions.
     */
    protected static function renderWidgetTableRowActions(array $widget): string
    {
        $actions = [];
        $widget_id = $widget['id'] ?? '';
        $widget_widget_id = $widget['widget_id'] ?? '';
        $widget_type = $widget['type'] ?? '';

        // Edit action
        $actions['edit'] = sprintf(
            '<a href="%s" class="btn-icon" data-toggle="tooltip" title="%s"><i class="tour-guide-icon tour-guide-icon-pencil"></i></a>',
            tourGuideHelper()->adminUrl('widgets/edit/' . $widget_id),
            tourGuideHelper()->translate('tour_guide_edit')
        );

        // Embed code action
        $actions['embed'] = sprintf(
            '<button type="button" class="btn-icon tour-guide-show-embed" data-widget-id="%s" data-widget-type="%s" data-toggle="tooltip" title="%s"><i class="tour-guide-icon tour-guide-icon-code-slash"></i></button>',
            self::escapeHtml($widget_widget_id),
            self::escapeHtml($widget_type),
            tourGuideHelper()->translate('tour_guide_widget_embed_code')
        );

        // Delete action
        $actions['delete'] = sprintf(
            '<a href="%s" class="btn-icon text-danger _delete delete" data-toggle="tooltip" title="%s" onclick="return confirm(\'%s\')"><i class="tour-guide-icon tour-guide-icon-trash-can"></i></a>',
            tourGuideHelper()->adminUrl('widgets/delete/' . $widget_id),
            tourGuideHelper()->translate('tour_guide_delete'),
            tourGuideHelper()->translate('tour_guide_delete_confirm')
        );

        // Allow actions to be filtered/modified
        $actions = TourGuideHook::applyFilter('tour_guide_widget_table_actions', $actions, $widget);

        return '<div class="tw-flex tw-items-center tw-gap-1">' . implode('', $actions) . '</div>';
    }

    /**
     * Render embed code modal and JavaScript.
     */
    protected static function renderWidgetEmbedModal(): void
    {
        echo TourGuideUtils::renderView('widgets/embed_modal', []);
    }
}