<?php

namespace TourGuide\Widgets;

use TourGuide\TourGuideUtils;

class TourGuideWidgetUtils
{
    const WIDGET_FORM_FIELD_PREFIX = 'tour_guide_widget';

    /**
     * Get widget form fields configuration.
     *
     * @return array Widget form fields configuration.
     */
    public static function getWidgetFormFields()
    {
        $form_fields = [
            'title' => [
                'default' => '',
                'input_type' => 'text',
                'label' => tourGuideHelper()->translate('tour_guide_widget_title'),
                'extra' => ['maxlength' => 255, 'class' => 'form-control', 'required' => 'required']
            ],
            'description' => [
                'default' => '',
                'input_type' => 'textarea',
                'label' => tourGuideHelper()->translate('tour_guide_widget_description'),
                'extra' => ['maxlength' => 500, 'class' => 'form-control']
            ],
            'type' => [
                'default' => 'player',
                'input_type' => 'hidden',
                'label' => tourGuideHelper()->translate('tour_guide_widget_type'),
                'options' => [
                    'player' => tourGuideHelper()->translate('tour_guide_widget_type_player'),
                    'setup' => tourGuideHelper()->translate('tour_guide_widget_type_setup')
                ],
                'extra' => ['class' => 'form-control']
            ],
            'status' => [
                'default' => 'active',
                'input_type' => 'select',
                'label' => tourGuideHelper()->translate('tour_guide_status'),
                'options' => [
                    'active' => tourGuideHelper()->translate('tour_guide_active'),
                    'inactive' => tourGuideHelper()->translate('tour_guide_inactive')
                ],
                'extra' => ['class' => 'form-control']
            ],
            'selected_tours' => [
                'default' => [],
                'input_type' => 'select',
                'label' => tourGuideHelper()->translate('tour_guide_widget_selected_tours'),
                'label_hint' => tourGuideHelper()->translate('tour_guide_widget_selected_tours_hint'),
                'options' => [], // Will be populated dynamically
                'extra' => [
                    'class' => 'form-control selectpicker',
                    'multiple' => true,
                    'size' => '8'
                ]
            ],
            'domain_whitelist' => [
                'default' => '',
                'input_type' => 'textarea',
                'label' => tourGuideHelper()->translate('tour_guide_widget_allowed_domains'),
                'label_hint' => tourGuideHelper()->translate('tour_guide_widget_domain_whitelist_hint'),
                'extra' => [
                    'class' => 'form-control',
                    'placeholder' => 'example.com, *.mydomain.com, localhost',
                    'rows' => '3'
                ]
            ]
        ];

        $settings_form_fields = [];

        return [
            'default' => $form_fields,
            'settings' => $settings_form_fields
        ];
    }

    /**
     * Get widget form data from POST request.
     *
     * @return array Processed widget form data.
     */
    public static function getWidgetFormData()
    {
        $data = TourGuideUtils::getPostData(self::WIDGET_FORM_FIELD_PREFIX, true) ?? [];
        $settings = $data['settings'] ?? [];

        $form_fields = self::getWidgetFormFields();

        // Process default fields
        foreach ($form_fields['default'] as $name => $field) {
            if ($field['input_type'] === 'checkbox') {
                $data[$name] = isset($data[$name]) ? true : false;
            } else {
                $data[$name] = $data[$name] ?? $field['default'];
            }
        }

        // Process settings fields
        foreach ($form_fields['settings'] as $name => $field) {
            if ($field['input_type'] === 'checkbox') {
                $settings[$name] = isset($settings[$name]) ? true : false;
            } else {
                $settings[$name] = $settings[$name] ?? $field['default'];
            }
        }

        // Process selected tours
        $data['selected_tours'] = $data['selected_tours'] ?? [];
        if (is_string($data['selected_tours'])) {
            $data['selected_tours'] = array_filter(explode(',', $data['selected_tours']));
        }

        // Process domain whitelist
        $data['domain_whitelist'] = $data['domain_whitelist'] ?? [];
        if (is_string($data['domain_whitelist'])) {
            $domains = array_filter(array_map('trim', explode(',', $data['domain_whitelist'])));
            $data['domain_whitelist'] = $domains;
        }

        $data['settings'] = $settings;
        return $data;
    }

    /**
     * Render widget form using dynamic field rendering similar to tour guide form.
     *
     * @param array $widget Widget data (null for new widget)
     * @param array $toursOptions Available tours as options array
     * @return string Generated HTML markup for the form
     */
    public static function renderWidgetForm(array $widget = [], array $toursOptions = [], string $groupSeparator = '<hr class="tour-guide-section-divider">')
    {
        $fields = self::getWidgetFormFields();

        // Convert widget settings to flat structure if needed
        if (!empty($widget['settings']) && is_string($widget['settings'])) {
            $widget['settings'] = json_decode($widget['settings'], true);
        }

        // Populate tour options dynamically
        $fields['default']['selected_tours']['options'] = $toursOptions;

        // Apply filter to allow field customization
        $fields = \TourGuide\TourGuideHook::applyFilter('tour_guide_widget_form_fields', $fields);

        // Prepare domain whitelist value
        $domains = $widget['domain_whitelist'] ?? [];
        if (is_array($domains)) {
            $widget['domain_whitelist'] = implode(', ', $domains);
        } else if (is_string($domains)) {
            $domainsArray = json_decode($domains, true);
            $widget['domain_whitelist'] = is_array($domainsArray) ? implode(', ', $domainsArray) : $domains;
        }

        $form = '<div class="tour-guide-widget-form-wrapper">';

        // Basic Information Section
        $form .= '<div class="tour-guide-content-wrapper tour-guide-widget-basic">';

        foreach ($fields['default'] as $name => $field) {
            // Skip selected_tours if no tours available
            if ($name === 'selected_tours' && empty($toursOptions)) {
                continue;
            }

            $field['default'] = $widget[$name] ?? $field['default'] ?? '';
            $fieldName = self::WIDGET_FORM_FIELD_PREFIX . '[' . $name . ']';

            $form .= \TourGuide\TourGuideUIHelper::renderField($fieldName, $field);

            // Add hint
            if (isset($field['label_hint'])) {
                $form .= '<small class="form-text text-muted">' . $field['label_hint'] . '</small>';
            }

            // Add divider after certain fields
            if (in_array($name, ['status', 'selected_tours'])) {
                $form .= $groupSeparator;
            }
        }

        // Show no tours message if needed
        if (empty($toursOptions)) {
            $form .= '<p class="tour-guide-no-tours">' .
                tourGuideHelper()->translate('tour_guide_no_tours_available') .
                '</p>';
        }

        $form .= '</div>';

        // Widget Settings Section
        if (!empty($fields['settings'])) {
            $form .= $groupSeparator;

            $form .= '<div class="tour-guide-content-wrapper tour-guide-widget-settings">';

            foreach ($fields['settings'] as $name => $field) {
                $field['label'] = tourGuideHelper()->translate('tour_guide_widget_setting_' . $name);
                $field['default'] = $widget['settings'][$name] ?? $field['default'] ?? '';
                $fieldName = self::WIDGET_FORM_FIELD_PREFIX . '[settings][' . $name . ']';

                $form .= \TourGuide\TourGuideUIHelper::renderField($fieldName, $field);
            }

            $form .= '</div>';
        }

        $form .= '</div>'; // Close wrapper

        // Apply filter to allow full form customization
        $form = \TourGuide\TourGuideHook::applyFilter('tour_guide_widget_form_output', $form, $widget, $toursOptions);

        return $form;
    }


    /**
     * Validate domain against whitelist.
     *
     * @param string $domain Domain to validate.
     * @param array $whitelist Array of allowed domains.
     * @return bool True if domain is allowed.
     */
    public static function validateDomain($domain, $whitelist = [])
    {
        if (empty($whitelist)) {
            return true; // No restrictions
        }

        // Remove protocol and www
        $domain = preg_replace('/^(https?:\/\/)?(www\.)?/', '', $domain);
        $domain = rtrim($domain, '/');

        foreach ($whitelist as $allowedDomain) {
            $allowedDomain = preg_replace('/^(https?:\/\/)?(www\.)?/', '', $allowedDomain);
            $allowedDomain = rtrim($allowedDomain, '/');

            // Support wildcards
            if (strpos($allowedDomain, '*') !== false) {
                $escaped = preg_quote($allowedDomain, '/');
                $escaped = str_replace('\*', '.*', $escaped);
                $pattern = '/^' . $escaped . '$/i';
                if (preg_match($pattern, $domain)) {
                    return true;
                }
            } elseif (strcasecmp($domain, $allowedDomain) === 0) {
                return true;
            }
        }

        return false;
    }

    /**
     * Initialize widget - adapted from TourGuideUIHelper::init().
     *
     * @param array $widgetConfig Widget configuration.
     * @param string $baseUrl Base URL for assets.
     * @return array Initialization data including scripts, templates, and URLs.
     */
    public static function initWidget($widgetConfig, $baseUrl = '')
    {
        // Determine context based on widget type
        $context = ($widgetConfig['type'] ?? 'player') === 'setup' ? 'setup' : 'widget';

        // Prepare URLs and tours data
        $assetUrl = tourGuideHelper()->asset('');
        $apiUrl = rtrim($baseUrl, '/') . '/api';

        // Prepare tours data for the widget
        $toursData = [];
        if (!empty($widgetConfig['tours'])) {
            foreach ($widgetConfig['tours'] as $tour) {
                $toursData[$tour->id] = (array)$tour;
            }
        }

        // Get user data from integration context (URL params, integration source, or fallback)
        $user = self::getWidgetUserData();

        $initData = [
            'commonScriptContent' => '',
            'setupTemplatesContent' => '',
            'context' => $context,
            'assetUrl' => $assetUrl,
            'apiUrl' => $apiUrl,
            'toursData' => $toursData,
            'active_user' => $user,
        ];

        try {
            $tour_guide_id = ($context === 'setup') ? 1 : null; // Dummy ID for setup widgets
            $config = [
                'context' => $context,
                'custom_asset_loading' => true,
                'load_as_fragment' => true,
                'active_user' => $user,
            ];

            // Get setup templates if it's a setup widget
            if (!empty($tour_guide_id)) {
                // @todo make Authentication before doing this.
                $initData['setupTemplatesContent'] = \TourGuide\TourGuideUtils::renderView('setup', [
                    'active_tour_guide_id' => $tour_guide_id,
                    'tour_guides' => $toursData,
                    'config' => $config
                ], true);
            }

            // Get the common script content using existing view
            $initData['commonScriptContent'] =  \TourGuide\TourGuideUtils::renderView('scripts/common', [
                'tour_guides' => $toursData,
                'config' => $config
            ], true);
        } catch (\Throwable $th) {
            error_log('Widget initialization error: ' . $th->getMessage());
        }

        return $initData;
    }

    /**
     * Generate widget JavaScript for embedding.
     *
     * @param array $widgetConfig Widget configuration.
     * @param string $baseUrl Base URL for assets.
     * @return string Generated JavaScript code.
     */
    public static function generateWidgetJS($widgetConfig, $baseUrl = '')
    {
        // Initialize widget using dedicated initWidget method
        $initData = self::initWidget($widgetConfig, $baseUrl);
        $initData['baseUrl'] = $baseUrl;
        $initData['widgetConfig'] = $widgetConfig;

        // Render the widget JavaScript using view file
        $widgetScript = trim(\TourGuide\TourGuideUtils::renderView('widgets/embed_script', $initData, true));

        // Remove script tags if present (added for IDE syntax highlighting)
        $widgetScript = preg_replace('/^<script>\s*/', '', $widgetScript);
        $widgetScript = preg_replace('/\s*<\/script>\s*$/', '', $widgetScript);

        return trim($widgetScript);
    }

    /**
     * Get user data for widget initialization.
     * Checks for external user data from URL parameters, then falls back to integration source.
     *
     * @return array User data array.
     */
    public static function getWidgetUserData()
    {
        // Check for external user data via URL parameters (for embedded widgets)
        $getData = \TourGuide\TourGuideUtils::getGetData(null, true);

        $externalUserId = $getData['user_id'] ?? '';
        if (!empty($externalUserId)) { // prefix with widget user i.e wu
            $externalUserId = 'wu-' . $externalUserId;
        }

        $externalUserEmail = $getData['user_email'] ?? '';
        $externalUserFirstName = $getData['user_first_name'] ?? '';
        $externalUserLastName = $getData['user_last_name'] ?? '';
        // Validate role against available roles, default to 'Guest'
        $providedRole = $getData['user_role'] ?? '';
        $availableRoles = array_map('strtolower', tourGuideHelper()->userRoles());
        $externalUserRole = in_array(strtolower($providedRole), $availableRoles) ? $providedRole : 'Guest';
        $externalUserLocale = $getData['user_locale'] ?? 'en';

        // If external user data provided, use it and fetch metadata if user ID exists
        if (!empty($externalUserId) || !empty($externalUserEmail)) {
            $userMetadata = [];

            // Try to get user metadata from the tour guide system if user ID provided
            if (!empty($externalUserId)) {
                try {
                    $userMetadata = (array)tourGuideHelper()->getRepository()->getUserMetadata($externalUserId);
                } catch (\Throwable $th) {
                    // Metadata fetch failed, continue with empty metadata
                }
            }

            return [
                'id' => $externalUserId ?: $externalUserEmail,
                'role' => $externalUserRole,
                'first_name' => $externalUserFirstName,
                'last_name' => $externalUserLastName,
                'email' => $externalUserEmail,
                'finished_tours' => $userMetadata['finished_tours'] ?? [],
                'last_view' => $userMetadata['last_view'] ?? [],
                'locale' => $externalUserLocale,
            ];
        }

        // Try to get user from integration source (when running within the app)
        try {
            return tourGuideHelper()->activeUser();
        } catch (\Throwable $th) {
            // Fallback for external embedding where no user context exists
            return [
                'role' => 'Guest',
                'id' => 'guest',
                'first_name' => '',
                'email' => '',
                'finished_tours' => [],
                'last_name' => '',
                'last_view' => [],
                'locale' => 'en',
            ];
        }
    }

    /**
     * Minify JavaScript code.
     *
     * @param string $js JavaScript code to minify.
     * @return string Minified JavaScript.
     */
    public static function minifyJS($js)
    {
        //@todo implement js minifier
        return trim($js);
    }
}